import {
  AlignItems,
  Background,
  Column,
  CoreText,
  Display,
  FlexDirection,
  Grid,
  GridGutterSize,
  Layout,
  StyledLayout,
  SVG,
  SVGAsset,
  SVGType,
  Tower,
  TowerChildWidth,
} from 'carbon-components-prototype';
import * as React from 'react';
import { RouteComponentProps } from 'react-router-dom';
import { ContributeCallout } from '../components/contribute-callout';
import { Markdown } from '../components/markdown';
import { PageWrapper } from '../components/page-wrapper';
import { TableOfContents } from '../components/table-of-contents';
import { File, FileConnection, SitePage, TypeDocFile } from '../graphql-types';
import './styles.scss';

interface PublicProps {
  data: {
    indexPage: SitePage;
    indexFile: File;
    tabs: FileConnection;
    activeTab: File;
    component?: TypeDocFile;
  };
}

type Props = PublicProps & RouteComponentProps<{}>;

// Gatsby requires a default export for template and layout files.
// tslint:disable-next-line:no-default-export
export default class PageTemplate extends React.Component<Props, {}> {
  public render () {
    return (
      <div>
        <PageWrapper
          page={this.props.data.indexPage}
          file={this.props.data.indexFile}
          tabs={this.props.data.tabs}
          component={this.props.data.component}
        >
          {this.props.data.activeTab.childMarkdownRemark && this.props.data.activeTab.childMarkdownRemark.html && (
            <Grid gutterSize={GridGutterSize.Large}>
              <Column cols={{ default: 12, lg: 8, xl: 7, xxl: 6 }}>
                <Layout>
                  <Markdown source={this.props.data.activeTab.childMarkdownRemark.html} />

                  {this.props.history.location.pathname.match(/components\/svg/) && this.renderIconList()}

                  <Layout margin={{ top: 4 }}>
                    <ContributeCallout
                      currentPageRelativePath={`data/${this.props.data.activeTab.relativePath}`}
                    />
                  </Layout>
                </Layout>
              </Column>
              <Column offset={{ default: 0, xl: 1 }} cols={{ default: 12, lg: 4, xl: 3 }}>
                <Layout padding={{ y: 2 }} className="default-page__table-of-contents">
                  {
                    this.props.data.activeTab.childMarkdownRemark &&
                    this.props.data.activeTab.childMarkdownRemark.headings && (
                      <TableOfContents headings={this.props.data.activeTab.childMarkdownRemark.headings} />
                    )
                  }
                </Layout>
              </Column>
            </Grid>
          )}
        </PageWrapper>
      </div>
    );
  }

  private renderIconList = () => {
    return (
      <Layout padding={1}>
        <Tower childWidth={TowerChildWidth.ExtraSmall} placeholderItems={20}>
          {Object.keys(SVGAsset).filter((icon) => !+icon).map((icon, index) => {
            const asset: SVGAsset = SVGAsset[icon as keyof typeof SVGAsset];
            return (
              <Layout padding={0.5} key={`icon-${index}`}>
                <StyledLayout
                  background={Background.Alt}
                  padding={1}
                  display={Display.Flex}
                  flexDirection={FlexDirection.Column}
                  alignItems={AlignItems.Center}
                >
                  <Layout padding={{ y: 1 }}>
                    <SVG type={SVGType.Brand} asset={asset} width={20} height={20} />
                  </Layout>
                  <CoreText ellipsis>{icon}</CoreText>
                </StyledLayout>
              </Layout>
            );
          })}
        </Tower>
      </Layout>
    );
  }
}

export const pageQuery = graphql`
  query DefaultPageContent(
    $pagePath: String!,
    $componentId: String
    $path: String,
    $tab: String,
  ) {
    ...defaultPage
    activeTab: file (
      fields: {
        path: { eq: $pagePath }
        tab: {
          slug: { eq: $tab }
        }
      }
    ) {
      relativePath
      fields {
        tabPath
        tab {
          title
          slug
        }
      }
      childMarkdownRemark {
        html
        headings {
          value
          depth
        }
      }
    }
  }
`;
