const chalk = require('chalk');
const fs = require('fs');
const glob = require('glob');
const path = require('path');

const baseDir = path.resolve(__dirname, '..');

const args = process.argv.slice(2);
const lintExistence = args.includes('--exists');

process.on('exit', (code) => {
  if (code < 1) {
    console.log('No issues found.');
  }
});

function error(text, lineNum) {
  const line = chalk.red.bold(`${lineNum}: `);
  const message = chalk.red(text);
  console.log(`${line}${message}`);
  process.exitCode = 1;
}

async function syncGlob(pattern) {
  return new Promise((resolve, reject) => {
    try {
      glob(path.join(baseDir, pattern), (err, files) => {
        if (files && files.length > 0) {
          resolve(files);
        } else {
          reject(err);
        }
      });
    } catch (err) {
      reject(err);
    }
  });
}

// Makes sure we don't try to lint a comment/blank line
function lineIsCodeownerEntry(line) {
  return !line.startsWith('#') && line.trim().length > 0;
}


// Validates that any 2 subsequent lines are ordered correctly
function validateOrdering(nextPath, prevPath, lineNum) {
  if (!prevPath) {
    return;
  }

  // Validate alphabetical ordering
  if (prevPath.toLowerCase() >= nextPath.toLowerCase()) {
    error(`Entries must be alphabetized by path. ${chalk.bold(nextPath)} should be before ${chalk.bold(prevPath)}`, lineNum);
  }
}

// Validates a dir ends with /, and a path does not
async function validatePath(linePath, lineNum) {
  // If --exists was used as an argument, verify the path exists on disk
  if (lintExistence) {
    try {
      await syncGlob(linePath);
    } catch (err) {
      error(`No file or directory exists at ${chalk.bold(linePath)}`, lineNum);
      return;
    }
  }

  const pathParts = path.parse(linePath);
  if (
    !pathParts.ext
    && linePath.slice(-1) !== '/'
    && !linePath.startsWith('/.') // top-level dotfiles should not capture here but count as an ext
    && linePath !== '*' // unowned, don't capture
  ) {
    // This is a dir, let's force a / at the end
    error(`${chalk.bold(linePath)} is a directory, and should end with /`, lineNum);
  } else if (
    pathParts.ext
    && linePath.slice(-1) === '/'
  ) {
    // This is a file, there should not be a / at the end
    error(`${chalk.bold(linePath)} is a file, and should not end with /`, lineNum);
  }

  // * exception for unowned
  if (!linePath.startsWith('/') && linePath !== '*') {
    error(`${chalk.bold(linePath)} should begin with /`, lineNum);
  }
}

function lint() {
  console.log(chalk.bold('Linting CODEOWNERS'));

  const fileText = fs.readFileSync(path.join(baseDir, '.github', 'CODEOWNERS'), 'utf-8');
  const lines = fileText.split('\n');

  lines.forEach(async (line, i) => {
    if (!lineIsCodeownerEntry(line)) {
      return;
    }

    // Lines are not 0-indexed
    const lineNum = i + 1;

    // Get pieces
    const prevLine = lines[i - 1];
    const prevLinePath = lineIsCodeownerEntry(prevLine) && prevLine.split(' ')[0];
    const linePieces = line.split(' ');
    const linePath = linePieces[0];
    const teams = linePieces.slice(1);

    await validatePath(linePath, lineNum);
    validateOrdering(linePath, prevLinePath, lineNum);
  });
}

lint();
