import * as React from 'react';
import { Align, Aspect, AspectRatio } from '../../aspect';
import { CoreImage, CoreImageProps } from '../../core-image';
import { Color, CoreText, FontSize, TextType } from '../../core-text';
import { Interactable, InteractableType } from '../../interactable';
import { AlignItems, Display, FlexDirection, InjectLayout, JustifyContent, Layout, Overflow, PaddingValue, Position, ZIndex } from '../../layout';
import { Background, StyledLayout } from '../../styled-layout';
import { SVG, SVGAsset } from '../../svg';
import './styles.scss';

export interface AccordionHeaderProps {
  index?: number;
  /**
   * @example What is thy bidding?
   */
  title: string;
  /**
   * @example Reach 10 followers
   */
  description?: string;
  isOpen?: boolean;
  onClick?: (event: React.MouseEvent<HTMLElement>, index?: number) => void;
  imageProps?: CoreImageProps;
  backgroundImageProps?: CoreImageProps;
  imageLabelOverlay?: string;
  label?: string;
}

/**
 * An AccordionHeader is an interactable region which should be used to toggle
 * the visibility of additional information.
 */
export class AccordionHeader extends React.Component<AccordionHeaderProps, {}> {
  public render() {
    return (
      <StyledLayout
        className="accordion-header"
        background={Background.Base}
        elevation={1}
        position={Position.Relative}
        overflow={Overflow.Hidden}
      >

        {this.renderBackgroundImage()}

        <Interactable
          onClick={this.onClick}
          type={InteractableType.Base}
          blurAfterClick
          aria-expanded={this.props.isOpen}
        >

          <Layout
            display={Display.Flex}
            position={Position.Relative}
            zIndex={ZIndex.Above}
          >

            <Layout flexGrow={0} flexShrink={0} display={Display.Flex} padding={{ y: this.getYPadding(), left: 2, right: 1 }}>
              <SVG asset={this.props.isOpen ? SVGAsset.AngleDown : SVGAsset.AngleUp} />
            </Layout>

            {this.renderTitles()}

            {this.renderLabel()}

            {this.renderImageAndLabel()}

          </Layout>

        </Interactable>

      </StyledLayout>
    );
  }

  private getYPadding = (): PaddingValue => {
    return this.props.description ? 2 : 3;
  }

  private renderTitles = () => {
    return (
      <Layout
        flexGrow={1}
        flexShrink={1}
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        alignItems={AlignItems.Start}
        justifyContent={JustifyContent.Center}
        overflow={Overflow.Hidden}
        padding={{ y: this.getYPadding(), right: 2 }}
      >
        <Layout fullWidth>
          <CoreText type={TextType.H3} fontSize={FontSize.Size4} bold ellipsis>
            {this.props.title}
          </CoreText>
          {this.props.description && (
            <CoreText color={Color.Alt2} ellipsis>
              {this.props.description}
            </CoreText>
          )}
        </Layout>
      </Layout>
    );
  }

  private renderBackgroundImage = () => {
    if (!this.props.backgroundImageProps) {
      return;
    }

    return (
      <InjectLayout
        className="accordion-header__background-image"
        position={Position.Absolute}
        attachTop
        attachLeft
        fullWidth
        fullHeight
      >
        <CoreImage {...this.props.backgroundImageProps} />
      </InjectLayout>
    );
  }

  private renderLabel = () => {
    if (!this.props.label) {
      return;
    }

    return (
      <Layout
        padding={{ right: 2 }}
        flexGrow={1}
        flexShrink={0}
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.End}
      >
        <CoreText>{this.props.label}</CoreText>
      </Layout>
    );
  }

  private renderImageAndLabel = () => {
    if (!this.props.imageProps && !this.props.imageLabelOverlay) {
      return;
    }

    return (
      <StyledLayout
        className="accordion-header__image-container"
        flexGrow={0}
        flexShrink={0}
        background={Background.Alt2}
        position={Position.Relative}
      >
        {this.props.imageLabelOverlay && (
          <StyledLayout
            position={Position.Relative}
            fullHeight
            zIndex={ZIndex.Above}
            display={Display.Flex}
            alignItems={AlignItems.Center}
            justifyContent={JustifyContent.Center}
            padding={1}
            color={this.props.imageProps ? Color.Overlay : Color.Inherit}
            background={this.props.imageProps ? Background.Overlay : undefined}
          >
            <CoreText
              fontSize={FontSize.Size3}
              bold
            >
              {this.props.imageLabelOverlay}
            </CoreText>
          </StyledLayout>
        )}
        {this.props.imageProps && (
          <Layout
            position={Position.Absolute}
            fullWidth
            fullHeight
            attachTop
            attachLeft
          >
            <Aspect ratio={AspectRatio.Aspect1x1} align={Align.Center}>
              <CoreImage {...this.props.imageProps} />
            </Aspect>
          </Layout>
        )}
      </StyledLayout>
    );
  }

  private onClick = (event: React.MouseEvent<HTMLElement>) => {
    if (this.props.onClick !== undefined) {
      this.props.onClick(event, this.props.index);
    }
  }
}
