import * as React from 'react';
import { AriaRole } from '../../aria/aria';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { ButtonProps } from '../button/button';
import { Button, ButtonSize, ButtonType } from '../button/button';
import { ButtonIcon } from '../button/button-icon';
import { CoreLink, CoreLinkProps, CoreLinkType } from '../core-link';
import { Color, CoreText, FontSize, TextType } from '../core-text';
import { Column } from '../grid/column';
import { Grid } from '../grid/grid';
import { AlignItems, Display, FlexDirection, InjectLayout, JustifyContent, Layout, Position } from '../layout';
import { ResponsiveWrapper } from '../responsive-wrapper';
import { StyledLayout } from '../styled-layout';
import { SVG, SVGAsset } from '../svg';
import './styles.scss';

export interface AlertBannerProps {
  /**
   * Up to two actionable buttons as part of the banner.
   */
  actions?: [ButtonProps, ButtonProps | undefined];
  /**
   * Translated string for Close button (x) ariaLabel.
   */
  closeButtonAriaLabel: string;
  /**
   * Optional link CTA for further context.
   */
  link?: CoreLinkProps;
  /**
   * A brief description of state of the alert taking place.
   * @example The System is Down.
   */
  status?: string;
  /**
   * Alert message text.
   * @example Please stand by, we will inform you when it is back up.
   */
  message: string;
  /**
   * The type of alert
   * @example Error
   */
  type: AlertBannerType;
  /**
   * MouseEvent handler to handle a user clicking on the Close button (x)
   */
  onCloseButtonClick: (e: React.MouseEvent<HTMLElement>) => void;
}

export enum AlertBannerType {
  Error = 'tw-alert-banner--error',
  Info = 'tw-alert-banner--info',
  Success = 'tw-alert-banner--success',
  Warning = 'tw-alert-banner--warning',
}

const ALERT_BANNER_ICON: {[key: string]: SVGAsset} = {
  [AlertBannerType.Error]: SVGAsset.NotificationError,
  [AlertBannerType.Info]: SVGAsset.NotificationInfo,
  [AlertBannerType.Success]: SVGAsset.NotificationSuccess,
  [AlertBannerType.Warning]: SVGAsset.NotificationWarning,
};

export const AlertBanner: React.SFC<AlertBannerProps> = (props) => {
  const classes: ClassValue = {
    'tw-alert-banner': true,
    [props.type]: true,
  };

  let link: JSX.Element | undefined;
  let actions: JSX.Element[] = [];

  if (props.link) {
    link = (
      <InjectLayout margin={{ left: 1 }}>
        <CoreLink className="tw-alert-banner__link" to={props.link.to} type={CoreLinkType.Inherit} underline hoverColorInherit>{props.link.children}</CoreLink>
      </InjectLayout>
    );
  }

  if (props.actions) {
    props.actions.forEach((action, i) => {
      actions.push((
        <Layout key={i} margin={{ left: 1 }}>
          <Button {...action} size={ButtonSize.Small} overlay type={i === 0 ? ButtonType.Hollow : ButtonType.Text} />
        </Layout>
      ));
    });

  }

  return (
    <StyledLayout
      {...getDataProps(props)}
      role={AriaRole.Alert}
      className={cn('tw-alert-banner', classes)}
      fontSize={FontSize.Size6}
      color={Color.Overlay}
      alignItems={AlignItems.Center}
      position={Position.Relative}
      display={Display.Flex}
      fullWidth
      padding={{ left: 1 }}
      breakpointLarge={{
        padding: { x: 4 },
      }}
    >
      <ResponsiveWrapper centered>
        <Grid justifyContent={JustifyContent.Center}>
          <Column cols={{ default: 12, lg: 9, xl: 8, xxl: 7 }}>
            <Layout
              display={Display.Flex}
              alignItems={AlignItems.Center}
              position={Position.Relative}
            >
              <SVG asset={ALERT_BANNER_ICON[props.type]} width={20} height={20}/>
              <Layout display={Display.Flex} alignItems={AlignItems.Center} flexGrow={1}>
                {props.status &&
                  <InjectLayout margin={{ left: 1 }}>
                    <CoreText type={TextType.H2} fontSize={FontSize.Size6}>{props.status}</CoreText>
                  </InjectLayout>
                }
                {(props.message || link) &&
                  <Layout margin={{ left: 1 }}>
                    <CoreText type={TextType.P} fontSize={FontSize.Size6}>{props.message}{link}</CoreText>
                  </Layout>
                }
              </Layout>
              {actions &&
                <Layout
                  display={Display.Flex}
                  flexDirection={FlexDirection.RowReverse}
                  justifyContent={JustifyContent.End}
                >
                {actions}
                </Layout>
              }
              <Layout padding={{ x: 1 }}>
                <ButtonIcon overlay icon={SVGAsset.Close} ariaLabel={props.closeButtonAriaLabel} onClick={props.onCloseButtonClick} />
              </Layout>
            </Layout>
          </Column>
        </Grid>
      </ResponsiveWrapper>
    </StyledLayout>
  );
};

AlertBanner.displayName = 'AlertBanner';
