import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { InjectLayout } from '../layout';
import './styles.scss';

export enum AnimationDelay {
  Short = 'tw-animation--delay-short',
  Medium = 'tw-animation--delay-medium',
  Long = 'tw-animation--delay-long',
  ExtraLong = 'tw-animation--delay-extra-long',
}

export enum AnimationDuration {
  Short = 'tw-animation--duration-short',
  Medium = 'tw-animation--duration-medium',
  Long = 'tw-animation--duration-long',
  ExtraLong = 'tw-animation--duration-extra-long',
}

export enum AnimationFillMode {
  Both = 'tw-animation--fill-mode-both',
  None = 'tw-animation--fill-mode-none',
  Forwards = 'tw-animation--fill-mode-forwards',
  Backwards = 'tw-animation--fill-mode-backwards',
}

export enum AnimationTiming {
  Ease = 'tw-animation--timing-ease',
  Linear = 'tw-animation--timing-linear',
  EaseIn = 'tw-animation--timing-ease-in',
  EaseOut = 'tw-animation--timing-ease-out',
  EaseInOut = 'tw-animation--timing-ease-in-out',
}

export enum AnimationType {
  Bounce = 'tw-animation--bounce',
  BounceIn = 'tw-animation--bounce-in',
  BounceOut = 'tw-animation--bounce-out',
  FadeIn = 'tw-animation--fade-in',
  FadeOut = 'tw-animation--fade-out',
  SlideInTop = 'tw-animation--slide-in-top',
  SlideInRight = 'tw-animation--slide-in-right',
  SlideInBottom = 'tw-animation--slide-in-bottom',
  SlideInLeft = 'tw-animation--slide-in-left',
  SlideOutTop = 'tw-animation--slide-out-top',
  SlideOutRight = 'tw-animation--slide-out-right',
  SlideOutBottom = 'tw-animation--slide-out-bottom',
  SlideOutLeft = 'tw-animation--slide-out-left',
}

export interface AnimationProps {
  /**
   * The content to render inside the animation.
   * @example <CoreText fontSize={FontSize.Size4}>Toggle the "enabled" property to animate.</CoreText>
   */
  children?: React.ReactNode;
  /** The delay before the animation is triggered. */
  delay?: AnimationDelay;
  /** The duration of the animation. */
  duration?: AnimationDuration;
  /**
   * Used to trigger the animation. Toggling this prop will fire the animation.
   * @example true
   */
  enabled?: boolean;
  /** Applies the CSS `animation-fill-mode` property. */
  fillMode?: AnimationFillMode;
  /** When enabled, loop the animation. */
  loop?: boolean;
  /** Event called when the animation start. */
  onAnimationStart?: (e: React.AnimationEvent<HTMLDivElement>) => void;
  /** Event called when the animation ends. */
  onAnimationEnd?: (e: React.AnimationEvent<HTMLDivElement>) => void;
  /** The easing function that will be used for the animation. */
  timing?: AnimationTiming;
  /** The type of animation that will occur. */
  type: AnimationType;
}

/**
 * Animations provide context and cues to the user interface.
 */
export const Animation: React.SFC<AnimationProps> = (props) => {
  const classes: ClassValue = {
    'tw-animation': true,
    'tw-animation--animate': props.enabled,
    'tw-animation--loop': props.loop,
  };

  if (props.delay) {
    classes[props.delay] = true;
  }

  if (props.duration) {
    classes[props.duration] = true;
  } else {
    classes[AnimationDuration.Short] = true;
  }

  if (props.fillMode) {
    classes[props.fillMode] = true;
  } else {
    classes[AnimationFillMode.Both] = true;
  }

  if (props.timing) {
    classes[props.timing] = true;
  } else {
    classes[AnimationTiming.Ease] = true;
  }

  if (props.type) {
    classes[props.type] = true;
  }

  return (
    <InjectLayout className={cn(classes)} {...getDataProps(props)}>
      <div
        onAnimationStart={props.onAnimationStart}
        onAnimationEnd={props.onAnimationEnd}
        data-a-target="tw-animation-target"
      >
        {props.children}
      </div>
    </InjectLayout>
  );
};

Animation.displayName = 'Animation';
