import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import './styles.scss';

export enum AspectRatio {
  Aspect21x9 = 'tw-aspect--21x9',
  Aspect16x9 = 'tw-aspect--16x9',
  Aspect4x3 = 'tw-aspect--4x3',
  Aspect1x1 = 'tw-aspect--1x1',
  Aspect3x4 = 'tw-aspect--3x4',
  Aspect3x2 = 'tw-aspect--3x2',
  // Leaving BoxArt as a semantic alias for now.
  BoxArt = 'tw-aspect--3x4',
}

export enum Align {
  Top = 'tw-aspect--align-top',
  Center = 'tw-aspect--align-center',
  Bottom = 'tw-aspect--align-bottom',
}

export interface AspectProps {
  /** The aspect ratio to use. */
  ratio?: AspectRatio;
  /**
   * Align the content within the component (if the content has a different
   * aspect ratio than the component).
   */
  align?: Align;
  /**
   * The content to be contained within the aspect.
   *
   * @example <img src="https://static-cdn.jtvnw.net/s3_vods/twitchpresents/159083707/cf22b5ad-ce79-454b-bd91-c9831579ecfc/thumb/index-0000000000-1280x720.jpg" />
   */
  children?: React.ReactNode;
  /**
   * Show overflowing content (if the content has a different aspect ratio
   * than the component).
   */
  overflow?: boolean;
}

/**
 * Aspect is used to set the height of an element based on its width for
 * applications such as box art and thumbnail images.
 */
export const Aspect: React.SFC<AspectProps> = (props) => {
  const classes: ClassValue = {
    'tw-aspect': true,
    'tw-aspect--overflow': props.overflow,
  };

  if (props.ratio) {
    classes[props.ratio] = true;
  } else {
    classes[AspectRatio.Aspect16x9] = true;
  }

  if (props.align) {
    classes[props.align] = true;
  } else {
    classes[Align.Top] = true;
  }

  return (
    <figure className={cn(classes)} {...getDataProps(props)}>
      {props.children}
    </figure>
  );
};

Aspect.displayName = 'Aspect';
