import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { CoreImage, CoreImageSize, CoreSrcSet } from '../core-image';
import { AlignItems, Display, JustifyContent, Layout, Overflow, Position } from '../layout';
import { Presence, PresenceStatus } from '../presence';
import { BorderRadius, StyledLayout } from '../styled-layout';
import './styles.scss';

export const AVATAR_SIZE_CLASSNAME = 'tw-avatar--size-';

export type AvatarSize = 30 | 36 | 40 | 50 | 64 | 80 | 96 | 300;

export interface AvatarProps {
  /** The border radius of the avatar. */
  borderRadius?: BorderRadius;
  /**
   * The size of the avatar.
   *
   * @example 40
   */
  size: AvatarSize;
  /**
   * Alternative text for the image.
   *
   * @example Required alternative text.
   */
  alt: string;
  /** The source of the avatar image. */
  src?: string | null;
  /** A configuration object for responsive image display. Accepts an object with valid descriptors for keys and image urls for values. */
  srcSet?: CoreSrcSet;
  /** The intended display size of the image at various breakpoints. Accepts an array of objects that contain a size, and optionally a mediaCondition. */
  sizes?: CoreImageSize[];
  /** Event handler emitted if the image fails to load. */
  onError?: React.EventHandler<React.SyntheticEvent<HTMLImageElement>>;
  /** Event handler emitted when the image loads.  */
  onLoad?: React.EventHandler<React.SyntheticEvent<HTMLImageElement>>;
  /** Show or hide the presence indicator. */
  presenceIndicator?: boolean;
  /** Status of the channel represented. */
  presenceStatus?: PresenceStatus;
}

/**
 * Avatars are a visual represention channels on Twitch. They can also indicate
 * a channels presence — such as online or away.
 */
export const Avatar: React.SFC<AvatarProps> = (props) => {
  const classes: ClassValue = {
    'tw-avatar': true,
    [AVATAR_SIZE_CLASSNAME + props.size]: !!props.size,
  };

  let defaultAvatar: string;
  let indicator: JSX.Element | undefined;

  if (props.size === 300) {
    defaultAvatar = 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_300x300.png';
  } else if (props.size === 96 || props.size === 80) {
    defaultAvatar = 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_150x150.png';
  } else {
    defaultAvatar = 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png';
  }

  if (props.presenceIndicator) {
    indicator = (
      <Layout
        className="tw-avatar__presence"
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        alignItems={AlignItems.Center}
        position={Position.Absolute}
      >
        <Presence
          border
          status={props.presenceStatus}
        />
      </Layout>
    );
  }

  return (
    <figure
      className={cn(classes)}
      {...getDataProps(props)}
    >
      <StyledLayout
        borderRadius={props.borderRadius || BorderRadius.Medium}
        overflow={Overflow.Hidden}
      >
        <CoreImage
          className="tw-avatar__img"
          src={props.src || defaultAvatar}
          alt={props.alt}
          srcSet={props.srcSet}
          sizes={props.sizes}
          onLoad={props.onLoad}
          onError={props.onError}
        />
      </StyledLayout>
      {indicator}
    </figure>
  );
};

Avatar.displayName = 'Avatar';
