import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { AlignItems, Display, JustifyContent, Layout, Position } from '../layout';
import { BorderRadius, InjectStyledLayout, StyledLayout } from '../styled-layout';
import { SVG, SVGAsset } from '../svg';
import './styles.scss';

export enum BadgeType {
  Notification = 'tw-badge--notification',
  Alt = 'tw-badge--alt',
  Alt2 = 'tw-badge--alt-2',
  Brand = 'tw-badge--brand',
  Live = 'tw-badge--live',
  Alert = 'tw-badge--alert',
  Success = 'tw-badge--success',
  Prime = 'tw-badge--prime',
  Dashboard = 'tw-badge--dashboard',
}

export interface BadgeProps {
  /**
   * Sets a border around the badge that inherits the color of the
   * badges parent.
   */
  border?: boolean;
  /**
   * Icon within the badge.
   *
   * @example Roman1
   */
  icon: SVGAsset;
  /** Sets the fill color of the badge.  */
  type?: BadgeType;
  /**
   * Size of the badge.
   *
   * @example 40
   */
  size?: number;
}

/**
 * Badges can accompany information to provide a more easily-recognizable cue
 * to information.
 */
export const Badge: React.SFC<BadgeProps> = (props) => {
  const classes: ClassValue = {
    'tw-badge': true,
  };

  if (props.type) {
    classes[props.type] = true;
  }

  const styles = {
    width: props.size,
    height: props.size,
  };

  const badge = (
    <InjectStyledLayout
      display={Display.InlineFlex}
      justifyContent={JustifyContent.Center}
      alignItems={AlignItems.Center}
      borderRadius={BorderRadius.Rounded}
      className={cn(classes)}
      {...getDataProps(props)}
    >
      <div style={styles}>
        <Layout
          className="tw-badge__icon"
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          alignItems={AlignItems.Center}
        >
          <SVG
            asset={props.icon}
            fill
          />
        </Layout>
      </div>
    </InjectStyledLayout>
  );

  if (props.border) {
    return (
      <StyledLayout
        className="tw-badge__container"
        borderRadius={BorderRadius.Rounded}
        display={Display.InlineFlex}
        position={Position.Relative}
      >
        {badge}
      </StyledLayout>
    );
  } else {
    return badge;
  }
};

Badge.displayName = 'Badge';
