import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { Display, InjectLayout, Overflow, Position } from '../layout/index';
import { Background, BorderRadius, Elevation, StyledLayout } from '../styled-layout/index';
import './styles.scss';

export enum BalloonSize {
  ExtraSmall = 'tw-balloon--xs',
  Small = 'tw-balloon--sm',
  Medium = 'tw-balloon--md',
  Large = 'tw-balloon--lg',
  ExtraLarge = 'tw-balloon--xl',
}

export enum BalloonDirection {
  Top = 'tw-balloon--up',
  Bottom = 'tw-balloon--down',
  Left = 'tw-balloon--left',
  Right = 'tw-balloon--right',
  TopLeft = 'tw-balloon--up tw-balloon--left',
  TopRight = 'tw-balloon--up tw-balloon--right',
  BottomRight = 'tw-balloon--down tw-balloon--right',
  BottomLeft = 'tw-balloon--down',
  TopCenter = 'tw-balloon--up tw-balloon--center',
  BottomCenter = 'tw-balloon--down tw-balloon--center',
  LeftCenter = 'tw-balloon--left tw-balloon--center',
  RightCenter = 'tw-balloon--right tw-balloon--center',
}

export interface BalloonProps {
  /**
   * Content rendered in the Balloon
   *
   * @example <Layout padding={5} />
   */
  children?: React.ReactNode;
  /** Direction the balloon will go. */
  direction?: BalloonDirection;
  /** Apply elevation styling to the balloon. */
  elevation?: Elevation;
  /** Hide the pointer tail. */
  noTail?: boolean;
  /** Offsets the balloon's position horizontally; string should include a CSS unit such as '10px' */
  offsetX?: string;
  /** Offsets the balloon's position vertically; string should include a CSS unit such as '10px' */
  offsetY?: string;
  /**
   * Show the balloon. The Balloon's behavior is controlled by the
   * component that is using it.
   *
   * @example true
   */
  show?: boolean;
  /**
   * The width of the Balloon. A Balloon's height is determined by
   * its children.
   */
  size?: BalloonSize;
  /** Set the Background color of the tail to match the Background of the Balloon's contents */
  tailBackground?: Background;
  /** Number of pixels to shift the position of the Balloon's tail */
  tailOffset?: number;
}

/**
 * To facilitate a focused interface, important actions and information should
 * always be visible to the user; but, less-used actions and information can
 * be placed within Balloons and shown by clicking.
 */
export const Balloon: React.SFC<BalloonProps> = (props) => {
  const classes: ClassValue = {
    'tw-balloon': true,
  };

  if (props.size) {
    classes[props.size] = true;
  }

  if (props.direction) {
    classes[props.direction] = true;
  } else {
    classes[BalloonDirection.Bottom] = true;
  }

  let marginLeft: string | undefined;
  let marginRight: string | undefined;
  let marginTop: string | undefined;
  let marginBottom: string | undefined;

  if (props.offsetX) {
    if (
      props.direction === BalloonDirection.Left ||
      props.direction === BalloonDirection.TopLeft ||
      props.direction === BalloonDirection.BottomLeft ||
      props.direction === BalloonDirection.LeftCenter ||
      props.direction === BalloonDirection.BottomRight ||
      props.direction === BalloonDirection.TopRight
    ) {
      marginRight = props.offsetX;
    } else {
      marginLeft = props.offsetX;
    }
  }

  if (props.offsetY) {
    if (
      props.direction === BalloonDirection.Top ||
      props.direction === BalloonDirection.TopLeft ||
      props.direction === BalloonDirection.TopRight ||
      props.direction === BalloonDirection.TopCenter
    ) {
      marginBottom = props.offsetY;
    } else {
      marginTop = props.offsetY;
    }
  }

  const styles: React.CSSProperties = {
    marginLeft,
    marginRight,
    marginTop,
    marginBottom,
  };

  let offsetTop: string | undefined;
  let offsetRight: string | undefined;
  let offsetLeft: string | undefined;

  if (props.tailOffset) {
    if (
      props.direction === BalloonDirection.Left ||
      props.direction === BalloonDirection.Right ||
      props.direction === BalloonDirection.LeftCenter ||
      props.direction === BalloonDirection.RightCenter
    ) {
      offsetTop = `${props.tailOffset}px`;
    } else {
      if (props.direction === BalloonDirection.TopRight || props.direction === BalloonDirection.BottomRight) {
        offsetRight = `${props.tailOffset}px`;
        offsetLeft = 'auto';
      } else {
        offsetRight = 'auto';
        offsetLeft = `${props.tailOffset}px`;
      }
    }
  }

  const tailStyles: React.CSSProperties = {
    top: offsetTop,
    left: offsetLeft,
    right: offsetRight,
  };

  let tail: JSX.Element | undefined;

  if (!props.noTail) {
    tail = (
      <InjectLayout
        className="tw-balloon__tail"
        position={Position.Absolute}
        overflow={Overflow.Hidden}
      >
        <div style={tailStyles}>
          <StyledLayout
            className="tw-balloon__tail-symbol"
            position={Position.Absolute}
            background={props.tailBackground || Background.Base}
            border
          />
        </div>
      </InjectLayout>
    );
  }

  return (
    <InjectLayout
      display={props.show ? Display.Block : Display.Hide}
      position={Position.Absolute}
    >
      <div
        style={styles}
        className={cn(classes)}
        {...getDataProps(props)}
      >
        {tail}
        <StyledLayout
          border
          borderRadius={BorderRadius.Medium}
          background={Background.Base}
          elevation={props.elevation ? props.elevation : 1}
        >
          {props.children}
        </StyledLayout>
      </div>
    </InjectLayout>
  );
};

Balloon.displayName = 'Balloon';
