import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import { CoreInteractive, CoreInteractivePublicProps } from '../../core-interactive';
import { AlignItems, Display, JustifyContent, Layout } from '../../layout';
import { SVG, SVGAsset, SVGType } from '../../svg';
import './styles.scss';

export enum ButtonIconType {
  Default = '',
  Hollow = 'tw-button-icon--hollow',
  Primary = 'tw-button-icon--primary',
  Alert = 'tw-button-icon--alert',
  Secondary = 'tw-button-icon--secondary',
  Success = 'tw-button-icon--success',
}

export enum ButtonIconSize {
  Default = '',
  Small = 'tw-button-icon--small',
  Large = 'tw-button-icon--large',
}

const BUTTON_ICON_SIZES = {
  [ButtonIconSize.Small]: { width: 16, height: 16 },
  [ButtonIconSize.Default]: { width: 20, height: 20 },
  [ButtonIconSize.Large]: { width: 24, height: 24 },
};

export interface ButtonIconProps extends CoreInteractivePublicProps {
  dropdown?: boolean;
  overlay?: boolean;
  icon: SVGAsset;
  size?: ButtonIconSize;
  statusAlertIcon?: SVGAsset;
  type?: ButtonIconType;
}

export class ButtonIcon extends React.Component<ButtonIconProps, {}> {
  public render() {
    return (
      <CoreInteractive
        {...this.props}
        className={this.className}
        {...getDataProps(this.props)}
      >
        {this.buttonContents}
      </CoreInteractive>
    );
  }

  private get className() {
    const classes: ClassValue = {
      'tw-button-icon': true,
      'tw-button-icon--dropmenu': this.props.dropdown,
      'tw-button-icon--disabled': this.props.disabled,
      'tw-button-icon--overlay': this.props.overlay,
      'tw-button-icon--status': !!this.props.statusAlertIcon,
    };

    if (this.props.statusAlertIcon) {
      classes[ButtonIconType.Success] = true;
    } else if (this.props.type) {
      classes[this.props.type] = true;
    }

    if (this.props.size) {
      classes[this.props.size] = true;
    }

    return cn(classes);
  }

  private get buttonContents() {
    return [
      this.icon,
      this.dropdown,
    ];
  }

  private get icon() {
    if (!this.props.icon) {
      return;
    }

    const iconSize = BUTTON_ICON_SIZES[this.props.size || ButtonIconSize.Default];

    if (this.props.statusAlertIcon) {
      return (
        <Layout className="tw-button-icon__icon tw-button-icon__icon--status" display={Display.Flex} key="tw-button-icon">
          <Layout className="tw-button-icon__primary-icon" display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center}>
            <SVG width={iconSize.width} height={iconSize.height} type={SVGType.Inherit} asset={this.props.icon} />
          </Layout>
          <Layout className="tw-button-icon__alert-icon" display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center}>
            <SVG width={iconSize.width} height={iconSize.height} type={SVGType.Inherit} asset={this.props.statusAlertIcon} />
          </Layout>
        </Layout>
      );
    }

    return (
      <span className="tw-button-icon__icon" key="tw-button-icon">
        <SVG width={iconSize.width} height={iconSize.height} type={SVGType.Inherit} asset={this.props.icon} />
      </span>
    );
  }

  private get dropdown() {
    if (!this.props.dropdown) {
      return;
    }

    return (
      <span className="tw-button-icon__icon tw-button-icon__icon--right" key="tw-button-dropdown">
        <SVG asset={SVGAsset.GlyphArrDown} type={SVGType.Inherit} />
      </span>
    );
  }
}
