import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import { CoreInteractive, CoreInteractivePublicProps } from '../../core-interactive';
import { AlignItems, Display, JustifyContent, Layout, Position } from '../../layout';
import { LoadingSpinner, SpinnerSize } from '../../loading-spinner';
import { SVG, SVGAsset, SVGType } from '../../svg';
import './styles.scss';

export enum ButtonType {
  Default = '',
  Hollow = 'tw-button--hollow',
  Text = 'tw-button--text',
  Alert = 'tw-button--alert',
  Success = 'tw-button--success',
}

export enum ButtonSize {
  Default = '',
  Small = 'tw-button--small',
  Large = 'tw-button--large',
}

export enum ButtonState {
  Default = '',
  Loading = 'tw-button--state-loading',
  Success = 'tw-button--state-success',
}

const BUTTON_ICON_SIZES = {
  [ButtonSize.Small]: { width: 16, height: 16 },
  [ButtonSize.Default]: { width: 20, height: 20 },
  [ButtonSize.Large]: { width: 24, height: 24 },
};

export interface ButtonProps extends CoreInteractivePublicProps {
  /**
   * The content rendered within the button. This includes the button text.
   * @example Chat
   */
  children?: React.ReactNode;
  /** Indicates the button will open a dropdown. */
  dropdown?: boolean;
  /** Sets the button to stretch across the width of its container. */
  fullWidth?: boolean;
  state?: ButtonState;
  /** Displays an icon in the button. */
  icon?: SVGAsset;
  /** Indicates that this button overlays on a darker background. */
  overlay?: boolean;
  /** Displays a price associated with the button action. */
  purchase?: string;
  /** Size of the button. */
  size?: ButtonSize;
  /** Icon to display when the status button is hovered. */
  statusAlertIcon?: SVGAsset;
  /**
   * Text to display when the status button is hovered. Setting this property
   * turns the button into a "status" button.
   */
  statusAlertText?: string;
  /** The type of button to display. */
  type?: ButtonType;
}

/**
 *  Buttons communicate that an action will happen when clicked. Best used as
 *  primary actions inside of another component such as a Modals, Cards, and/or
 *  forms. **Buttons should never be used for generic links.**
 */
export class Button extends React.Component<ButtonProps, {}> {
  public render() {
    const disabled =
      this.props.disabled
      || this.props.state === ButtonState.Loading
      || this.props.state === ButtonState.Success;

    return (
      <CoreInteractive
        {...this.props}
        className={this.className}
        disabled={disabled}
        {...getDataProps(this.props)}
      >
        {this.buttonContents}
      </CoreInteractive>
    );
  }

  private get className() {
    const classes: ClassValue = {
      'tw-button': true,
      'tw-button--dropmenu': this.props.dropdown,
      'tw-button--overlay': this.props.overlay,
      'tw-button--purchase': !!this.props.purchase,
      'tw-button--status': !!this.props.statusAlertText || !!this.props.statusAlertIcon,
      'tw-button--disabled': this.props.disabled,
      'tw-button--full-width': this.props.fullWidth,
    };

    if (this.props.statusAlertText || this.props.statusAlertIcon || this.props.state === ButtonState.Success) {
      classes[ButtonType.Success] = true;
    } else if (this.props.type) {
      classes[this.props.type] = true;
    }

    if (this.props.size) {
      classes[this.props.size] = true;
    }

    if (this.props.state) {
      classes[this.props.state] = true;
    }

    return cn(classes);
  }

  private get buttonContents() {
    return [
      this.icon,
      this.buttonText,
      this.loader,
      this.successCheck,
      this.dropdown,
      this.purchase,
    ];
  }

  private get icon() {
    if (!this.props.icon) {
      return;
    }

    const iconSize = BUTTON_ICON_SIZES[this.props.size || ButtonSize.Default];

    if (this.props.statusAlertIcon) {
      return (
        <Layout className={`tw-button__icon tw-button__icon--status ${this.props.children ? 'tw-button__icon--left' : ''}`} display={Display.Flex} key="tw-button-icon">
          <Layout className="tw-button__primary-icon" display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center}>
            <SVG width={iconSize.width} height={iconSize.height} type={SVGType.Inherit} asset={this.props.icon} />
          </Layout>
          <Layout className="tw-button__alert-icon" display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center}>
            <SVG width={iconSize.width} height={iconSize.height} type={SVGType.Inherit} asset={this.props.statusAlertIcon} />
          </Layout>
        </Layout>
      );
    }

    return (
      <span className={`tw-button__icon ${this.props.children ? 'tw-button__icon--left' : ''}`} key="tw-button-icon">
        <SVG width={iconSize.width} height={iconSize.height} type={SVGType.Inherit} asset={this.props.icon} />
      </span>
    );
  }

  private get buttonText() {
    if (!this.props.children) {
      return;
    }

    if (this.props.statusAlertText) {
      return (
        <span className="tw-button__text" key="tw-button-text">
          <span className="tw-button__primary-text" data-a-target="tw-button-text">{this.props.children}</span>
          <span className="tw-button__alert-text">{this.props.statusAlertText}</span>
        </span>
      );
    }

    return (
      <span className="tw-button__text" data-a-target="tw-button-text" key="tw-button-text">
        {this.props.children}
      </span>
    );
  }

  private get loader() {
    if (this.props.state !== ButtonState.Loading) {
      return;
    }

    return (
      <Layout className="tw-button__loading-spinner" position={Position.Absolute} key="tw-button-loading-spinner">
        <LoadingSpinner delay={0} size={this.getSpinnerSize(this.props.size)} inheritColor />
      </Layout>
    );
  }

  private get successCheck() {
    if (this.props.state !== ButtonState.Success) {
      return;
    }

    return (
      <Layout className="tw-button__success-icon" position={Position.Absolute} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center} key="tw-button-success-icon">
        <SVG asset={SVGAsset.Check} width={this.successIconSize(this.props.size)} height={this.successIconSize(this.props.size)} />
      </Layout>
    );
  }

  private get dropdown() {
    if (!this.props.dropdown) {
      return;
    }

    return (
      <span className="tw-button__icon tw-button__icon--right" key="tw-button-dropdown">
        <SVG asset={SVGAsset.GlyphArrDown} type={SVGType.Inherit} />
      </span>
    );
  }

  private get purchase() {
    if (!this.props.purchase) {
      return;
    }

    return (
      <span className="tw-button__num-block" key="tw-button-purchase">
        {this.props.purchase}
      </span>
    );
  }

  private getSpinnerSize(size?: ButtonSize) {
    switch (size) {
      case ButtonSize.Small: {
        return SpinnerSize.Small;
      }
      case ButtonSize.Large: {
        return SpinnerSize.Large;
      }
      case ButtonSize.Default:
      default: {
        return SpinnerSize.Default;
      }
    }
  }

  private successIconSize(size?: ButtonSize) {
    switch (size) {
      case ButtonSize.Small: {
        return 16;
      }
      case ButtonSize.Large: {
        return 24;
      }
      case ButtonSize.Default:
      default: {
        return 20;
      }
    }
  }
}
