import * as React from 'react';
import { getDataProps } from '../../../utils/data-props';
import { Color } from '../../core-text';
import { AlignItems, Display, FlexDirection, FlexWrap, Layout, Position } from '../../layout';
import { Background, BorderRadius, Elevation, StyledLayout } from '../../styled-layout';

export interface CardProps {
  /** Display a border around the card. */
  border?: boolean;
  borderRadius?: BorderRadius;
  background?: Background;
  /**
   * The children.
   *
   * @example <CardImage
   *   src="https://static-cdn.jtvnw.net/ttv-boxart/League%20of%20Legends-285x380.jpg"
   *   alt="Leage of Legends"
   *   aspect={AspectRatio.Aspect3x4} />
   *   <CardBody>League of Legends</CardBody>
   */
  children?: React.ReactNode;
  /** Display an elevation shadow on the card. */
  elevation?: Elevation;
  /** Display the card as a row. */
  row?: boolean;
}

/**
 * Cards summarize information — such as games, channels, and communities — and
 * they provide a way of performing actions or viewing more detailed parts of
 * that information.
 */
export const Card: React.SFC<CardProps> = (props) => {
  let alignItems: AlignItems | undefined;
  let elevation: Elevation | undefined;
  let color: Color | undefined;

  if (props.row) {
    alignItems = AlignItems.Center;
  }

  if (props.elevation) {
    elevation = props.elevation;
  }

  if (props.background === Background.Overlay) {
    color = Color.Overlay;
  }

  return (
    <Layout
      className="tw-card"
      position={Position.Relative}
      {...getDataProps(props)}
    >
      <StyledLayout
        display={Display.Flex}
        flexDirection={props.row ? FlexDirection.Row : FlexDirection.Column}
        alignItems={alignItems}
        elevation={elevation}
        background={props.background}
        borderRadius={props.borderRadius}
        color={color}
        border={props.border}
        flexWrap={FlexWrap.NoWrap}
      >
        {props.children}
      </StyledLayout>
    </Layout>
  );
};

Card.displayName = 'Card';
