import * as React from 'react';
import { getDataProps } from '../../../utils/data-props';
import { getImageProps } from '../../../utils/image-props';
import { Aspect, AspectRatio } from '../../aspect';
import { Card } from '../../card/card';
import { CardBody } from '../../card/card-body';
import { CardImage } from '../../card/card-image';
import { CoreImageProps } from '../../core-image';
import { CoreInteractive, CoreInteractivePublicProps } from '../../core-interactive';
import { Color, CoreText, FontSize, LineHeight, TextType } from '../../core-text';
import { Display, Layout } from '../../layout';
import { Placeholder } from '../../placeholder/';
import { BorderRadius, StyledLayout } from '../../styled-layout';
import { Tag, TagProps, TagType } from '../../tag';
import './styles.scss';

export interface BoxArtCardProps extends CoreImageProps, CoreInteractivePublicProps {
  /** The meta information displayed under the title.
   * @example 106,303 viewers
   */
  info: string | JSX.Element;
  /** Show the placeholder representation of this card */
  placeholder?: boolean;
  /** A list of tag interfaces to render into tags underneath the card body */
  tags?: TagProps[];
  /** The card's title.
   * @example League of Legends
   */
  title: string;
  /** Displays the BoxArt as unavailable to launch */
  unavailable?: boolean;
}

/**
 *  Box art cards can be used to link to game and category views.
 */
export const BoxArtCard: React.SFC<BoxArtCardProps> = (props) => {

  let cardContent: JSX.Element;

  cardContent = (
    <>
      <Layout
        className={`tw-box-art-card__image ${props.unavailable && !props.disabled ? 'tw-box-art-card__image--unavailable' : '' }`}
        data-test-selector="tw-card-image"
        data-a-target="tw-card-image"
      >
        <CardImage
          aspect={AspectRatio.BoxArt}
          borderRadius={BorderRadius.Medium}
          {...getImageProps(props)}
        />
      </Layout>
      <CardBody>
        <Layout
          data-test-selector="tw-card-title"
          data-a-target="tw-card-title"
          margin={{ top: 0.5 }}
        >
          <CoreText
            className="tw-box-art-card__title"
            type={TextType.H3}
            fontSize={FontSize.Size5}
            lineHeight={LineHeight.Body}
            ellipsis
          >
            {props.title}
          </CoreText>
          <CoreText color={Color.Alt2} ellipsis>{props.info}</CoreText>
        </Layout>
      </CardBody>
    </>
  );

  if (props.placeholder) {
    return (
      <>
        <Layout margin={{ bottom: 0.5 }}>
          <Aspect ratio={AspectRatio.Aspect3x4}>
            <Placeholder />
          </Aspect>
        </Layout>
        <CoreText>
          <Placeholder width={150} />
        </CoreText>
        <CoreText fontSize={FontSize.Size7}>
          <Placeholder width={100} />
        </CoreText>
      </>
    );
  }

  return (
    <div className="tw-box-art-card" {...getDataProps(props)}>
      <Card key={props.title}>
        {props.linkTo || props.onClick || props.disabled ? (
          <CoreInteractive
            className={`tw-box-art-card__link ${props.disabled ? 'tw-box-art-card__link--disabled' : ''}`}
            data-a-target="tw-box-art-card-link"
            disabled={props.disabled}
            download={props.download}
            linkTo={props.linkTo}
            ariaLabel={props.title}
            blurAfterClick={props.blurAfterClick}
            onClick={props.onClick}
            renderLink={props.renderLink}
            targetBlank={props.targetBlank}
            tabIndex={props.tabIndex}
          >
            {cardContent}
          </CoreInteractive>
        ) : cardContent }
        { props.tags && (
          <Layout className="tw-box-art-card__tags" margin={{ top: 0.5 }}>
            {props.tags.map((tag, i) => (
              <StyledLayout key={i} fontSize={FontSize.Size7} display={Display.InlineBlock} margin={{ right: 0.5, bottom: 0.5 }}>
                <Tag {...tag} type={TagType.Default} data-a-target={tag.label}/>
              </StyledLayout>
            ))}
          </Layout>
        )}
      </Card>
    </div>
  );
};

BoxArtCard.displayName = 'BoxArtCard';
