import * as history from 'history';
import * as React from 'react';
import { Link } from 'react-router-dom';
import { getDataProps } from '../../../utils/data-props';
import { getImageProps } from '../../../utils/image-props';
import { Aspect, AspectRatio } from '../../aspect';
import { Card } from '../../card/card';
import { CardBody } from '../../card/card-body';
import { CardImage } from '../../card/card-image';
import { CoreImageProps } from '../../core-image';
import { Color, CoreText, FontSize, LineHeight, TextType } from '../../core-text';
import { Layout } from '../../layout';
import { Placeholder } from '../../placeholder';
import { BorderRadius } from '../../styled-layout';
import './styles.scss';

export interface ThumbnailCardProps extends CoreImageProps {
  /** The meta information displayed under the title. */
  info: string | JSX.Element;
  /** Make the card a link to the provided URL. */
  linkTo?: history.LocationDescriptor;
  /** Show the placeholder representation of this card */
  placeholder?: boolean;
  /** The card's title. */
  title: string;
}

export const ThumbnailCard: React.SFC<ThumbnailCardProps> = (props) => {
  let cardContent: JSX.Element;

  cardContent = (
    <Card key={props.title}>
      <CardImage
        aspect={AspectRatio.Aspect16x9}
        borderRadius={BorderRadius.Medium}
        {...getImageProps(props)}
      />
      <CardBody>
        <Layout margin={{ top: 0.5 }}>
          <CoreText
            className="tw-thumbnail-card__title"
            type={TextType.H3}
            fontSize={FontSize.Size5}
            lineHeight={LineHeight.Body}
            ellipsis
          >
            {props.title}
          </CoreText>
        </Layout>
        <CoreText color={Color.Alt2} ellipsis>{props.info}</CoreText>
      </CardBody>
    </Card>
  );

  if (props.placeholder) {
    return (
      <Layout>
        <Layout margin={{ bottom: 0.5 }}>
          <Aspect ratio={AspectRatio.Aspect16x9}>
            <Placeholder />
          </Aspect>
        </Layout>
        <CoreText>
          <Placeholder width={150} />
        </CoreText>
        <CoreText fontSize={FontSize.Size7}>
          <Placeholder width={100} />
        </CoreText>
      </Layout>
    );
  }

  if (props.linkTo) {
    return (
      <div className="tw-thumbnail-card" {...getDataProps(props)}>
        <Link to={props.linkTo} title={props.title} data-a-target="tw-thumbnail-card-link">
          {cardContent}
        </Link>
      </div>
    );
  } else {
    return (
      <div className="tw-thumbnail-card" {...getDataProps(props)}>
          {cardContent}
      </div>
    );
  }
};

ThumbnailCard.displayName = 'ThumbnailCard';
