import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';

// https://github.com/bitjourney/react-simple-image

const descriptorMatcher = (key: string) => {
  return new RegExp (/^(\d+\.?\d*)(w|x)$/).test(key);
};

export interface CoreImageSize {
  size: string;
  mediaCondition?: string;
}

export type CoreSrcSet = {
  [descriptor: string]: string;
};

export interface CoreImageProps {
  className?: string;
  /**
   * URL of the image resource.
   *
   * @example https://s3-us-west-2.amazonaws.com/web-design-int-production/assets/brand/images/Photography--1.jpg
   */
  src: string;
  /**
   * User-readable description of the image being displayed.
   *
   * @example Required alternative text.
   */
  alt: string;
  /**
   * A configuration object for responsive image display. Accepts an object with valid descriptors for keys and image urls for values.
   */
  srcSet?: CoreSrcSet;
  /**
   * The intended display size of the image at various breakpoints. Accepts an array of objects that contain a size, and optionally a mediaCondition.
   */
  sizes?: CoreImageSize[];
  /**
   * Callback that fires when image fails to load.
   */
  onError?: React.EventHandler<React.SyntheticEvent<HTMLImageElement>>;
  /**
   * Callback that fires when image loads.
   */
  onLoad?: React.EventHandler<React.SyntheticEvent<HTMLImageElement>>;

  /**
   * Should the image take up it's full height of it's parent
   */
  fullHeight?: boolean;

  /**
   * Should the image take up it's full width of it's parent
   */
  fullWidth?: boolean;

  refHandler?: (ref: HTMLImageElement | null) => void;
}

export const CoreImage: React.SFC<CoreImageProps> = (props) => {
  const classes: ClassValue = {
    'tw-image': true,
    'tw-full-height': props.fullHeight,
    'tw-full-width': props.fullWidth,
  };

  let sizes;

  if (props.sizes) {
    sizes = props.sizes.map((size) => {
      if (size.mediaCondition) {
        return `${size.mediaCondition} ${size.size}`;
      }
      return `${size.size}`;
    }).join(',') || undefined;
  }

  const srcSet = Object.keys(props.srcSet || {})
    .filter(descriptorMatcher)
    .map((descriptor) => props.srcSet ? `${props.srcSet[descriptor]} ${descriptor}` : '')
    .join(',') || undefined;

  // srcSet implementation is browser dependent, and in some browsers the attribute
  // order is important (especially in JS-generated HTML). Incorrect ordering
  // can cause the browser pre-loader to download multiple versions of an image,
  // especially with regard to w descriptor srcSets. The optimal ordering is
  // sizes > srcSet > src, as seen in the specification example:
  // https://html.spec.whatwg.org/multipage/images.html#viewport-based-selection
  return (
    <img
      className={cn(props.className, classes)}
      {...getDataProps(props)}
      alt={props.alt}
      sizes={sizes}
      srcSet={srcSet}
      src={props.src}
      onError={props.onError}
      onLoad={props.onLoad}
      ref={props.refHandler}
    />
  );
};

CoreImage.displayName = 'CoreImage';
