import * as classnames from 'classnames';
import * as history from 'history';
import * as React from 'react';
import { Link } from 'react-router-dom';
import { getDataProps } from '../../utils/data-props';
import { isExternalURL } from '../../utils/url-props';

/**
 * CoreInteractivePublicProps is what enclosing classes should extend in order
 * to have the full compliment of props for passing through to the
 * CoreInteractive component.
 */
export interface CoreInteractivePublicProps {
  /**
   * Sets the `aria-label` HTML attribute. This is label will be available
   * to screen readers.
   */
  ariaLabel?: string;
  /**
   * If the element is a button, autofocus it when it is first rendered.
   */
  autoFocus?: boolean;
  /**
   * Ensures the element loses focus after interaction.
   */
  blurAfterClick?: boolean;
  /** Sets the element as disabled, always renders as a button. */
  disabled?: boolean;
  /**
   * Sets the `download` attribute on the element. If the `linkTo` prop is set,
   * the resource will be downloaded instead of being navigated to.
   */
  download?: boolean | string;
  /** Indicates the element is a link and will navigate to the passed URL. */
  linkTo?: history.LocationDescriptor;
  /** Event called when the element is clicked. */
  onClick?: React.MouseEventHandler<HTMLElement>;
  /** Render prop to allow for using react-router alternatives @experimental */
  renderLink?: <P extends CoreInteractiveProps>(props: P) => React.ReactNode;
  /** Replace the current entry in the history stack instead of adding a new one. */
  replace?: boolean;
  /** Manually set the `tabIndex` of this element. */
  tabIndex?: number;
  /** If the element is a link, open it in a new tab. */
  targetBlank?: boolean;
  /** The elements title */
  title?: string;
}

/**
 * CoreInteractiveProps extends CoreInteractiveLinkProps, and enclosing
 * components should generate the things like className, refHandler, etc
 * internally and pass them down
 */
export interface CoreInteractiveProps extends CoreInteractivePublicProps {
  className?: string;
  /**
   * The content rendered within the element.
   */
  children?: React.ReactNode;
  refHandler?: (ref: HTMLElement | null) => void;
  role?: string;
}

export class CoreInteractive extends React.Component<CoreInteractiveProps, {}> {
  public render() {
    const target = this.props.targetBlank ? '_blank' : undefined;

    // links
    if (!this.props.disabled && this.props.linkTo) {
      // external link
      if (isExternalURL(this.props.linkTo)) {
        if (typeof this.props.linkTo !== 'string') {
          // Implication: typeof this.this.props.linkTo === LocationDescriptorObject
          throw new Error('External links with Location descriptors are not implemented! Pass a string instead.');
        }

        return (
          <a
            download={this.props.download}
            href={this.props.linkTo}
            ref={this.props.refHandler}
            target={target}
            title={this.props.title}
            {...this.baseProps}
          >
            {this.props.children}
          </a>
        );
      }

      // internal custom link using linkTo prop
      if (this.props.renderLink) {
        return this.props.renderLink(this.props);
      }

      // internal default link
      return (
        <Link
          download={this.props.download}
          replace={this.props.replace}
          to={this.props.linkTo}
          target={target}
          title={this.props.title}
          {...this.baseProps}
        >
          {this.props.children}
        </Link>
      );
    }

    // non-link or disabled
    return (
      <button
        autoFocus={this.props.autoFocus || undefined}
        disabled={this.props.disabled || undefined} // prevent disabled={false}
        ref={this.props.refHandler}
        title={this.props.title}
        {...this.baseProps}
      >
        {this.props.children}
      </button>
    );
  }

  private get baseProps() {
    return {
      'aria-label': this.props.ariaLabel,
      'className': classnames('tw-interactive', this.props.className),
      'onClick': this.onClick,
      'role': this.props.role,
      'tabIndex': this.props.tabIndex,
      ...getDataProps(this.props),
    };
  }

  private onClick = (e: React.MouseEvent<HTMLElement>) => {
    if (this.props.onClick) {
      this.props.onClick(e);
    }

    if (this.props.blurAfterClick && e && e.currentTarget) {
      e.currentTarget.blur();
    }
  }
}
