import * as React from 'react';
import { AriaProps } from '../../aria/aria';
import { getAriaProps } from '../../utils/aria-props';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { InjectLayout, VerticalAlign } from '../layout';
import './styles.scss';

export enum Color {
  Base = 'tw-c-text-base',
  Alt = 'tw-c-text-alt',
  Alt2 = 'tw-c-text-alt-2',
  Link = 'tw-c-text-link',
  Live = 'tw-c-text-live',
  Error = 'tw-c-text-error',
  Overlay = 'tw-c-text-overlay',
  OverlayAlt = 'tw-c-text-overlay-alt',
  Inherit = 'tw-c-text-inherit',
}

export enum FontSize {
  Size1 = 'font-size-1',
  Size2 = 'font-size-2',
  Size3 = 'font-size-3',
  Size4 = 'font-size-4',
  Size5 = 'font-size-5',
  Size6 = 'font-size-6',
  Size7 = 'font-size-7',
  Size8 = 'font-size-8',
}

export type Lines = 1 | 2;

export enum LineHeight {
  Body = 'tw-line-height-body',
  Heading = 'tw-line-height-heading',
}

export enum TextDecoration {
  Underline = 'tw-underline',
  Strikethrough = 'tw-strikethrough',
}

export enum TextTransform {
  Capitalize = 'tw-capcase',
  Uppercase = 'tw-upcase',
}

export enum TextType {
  P = 'p',
  Span = 'span',
  H1 = 'h1',
  H2 = 'h2',
  H3 = 'h3',
  H4 = 'h4',
  H5 = 'h5',
  H6 = 'h6',
  Strong = 'strong',
  Em = 'em',
}

export enum WordBreak {
  Normal = 'tw-word-break-normal',
  BreakAll = 'tw-word-break-all',
  KeepAll = 'tw-word-break-keep-all',
  BreakWord = 'tw-word-break-word',
}

export interface BreakpointProps {
  fontSize?: FontSize;
}

export interface CoreTextProps extends BreakpointProps, AriaProps {
  align?: VerticalAlign;
  bold?: boolean;
  breakpointExtraSmall?: BreakpointProps;
  breakpointSmall?: BreakpointProps;
  breakpointMedium?: BreakpointProps;
  breakpointLarge?: BreakpointProps;
  breakpointExtraLarge?: BreakpointProps;
  breakpointExtraExtraLarge?: BreakpointProps;
  children?: React.ReactNode;
  className?: string;
  color?: Color;
  decoration?: TextDecoration;
  ellipsis?: boolean;
  italic?: boolean;
  lineHeight?: LineHeight;
  lines?: Lines;
  noWrap?: boolean;
  strikethrough?: boolean;
  title?: string;
  transform?: TextTransform;
  type?: TextType;
  wordBreak?: WordBreak;
}

export class CoreText extends React.Component<CoreTextProps, {}> {
  public render() {
    const classes: ClassValue = {
      'tw-ellipsis': this.props.ellipsis || this.props.lines && this.props.lines > 0,
      'tw-italic': this.props.italic,
      'tw-nowrap': this.props.noWrap,
      'tw-strong': this.props.bold,
      [`tw-line-clamp-2`]: this.props.lines === 2,
    };

    if (this.props.color) {
      classes[this.props.color] = true;
    }

    if (this.props.lineHeight) {
      classes[this.props.lineHeight] = true;
    }

    if (this.props.transform) {
      classes[this.props.transform] = true;
    }

    if (this.props.decoration) {
      classes[this.props.decoration] = true;
    }

    if (this.props.wordBreak) {
      classes[this.props.wordBreak] = true;
    }

    const defaultBreakpoint = this.getBreakpointClasses(this.props);
    const extraSmallBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraSmall, 'xs');
    const smallBreakpoint = this.getBreakpointClasses(this.props.breakpointSmall, 'sm');
    const mediumBreakpoint = this.getBreakpointClasses(this.props.breakpointMedium, 'md');
    const largeBreakpoint = this.getBreakpointClasses(this.props.breakpointLarge, 'lg');
    const extraLargeBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraLarge, 'xl');
    const extraExtraLargeBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraExtraLarge, 'xxl');

    let title: string | undefined;
    if (this.props.title) {
      title = this.props.title;
    } else if (this.props.ellipsis && (typeof this.props.children === 'string')) {
      title = this.props.children;
    }

    let textElement = React.createElement<CoreTextProps, HTMLElement>(this.props.type || 'p', {
      className: cn(
        this.props.className,
        classes,
        defaultBreakpoint,
        extraSmallBreakpoint,
        smallBreakpoint,
        mediumBreakpoint,
        largeBreakpoint,
        extraLargeBreakpoint,
        extraExtraLargeBreakpoint,
      ),
      ...getDataProps(this.props),
      ...getAriaProps(this.props),
      title,
    }, this.props.children);

    return (
      <InjectLayout verticalAlign={this.props.align}>
        {textElement}
      </InjectLayout>
    );
  }

  private getBreakpointClasses(value?: CoreTextProps, prefix?: string) {
    if (!value) {
      return '';
    }

    let classes: string[] = [];

    if (prefix) {
      prefix = 'tw-' + prefix + '-';
    } else {
      prefix = 'tw-';
    }

    if (value.fontSize) {
      classes.push(prefix + value.fontSize);
    }

    return classes;
  }
}
