import * as React from 'react';
import { getImageProps } from '../../../utils/image-props';
import { Aspect, AspectRatio } from '../../aspect';
import { CoreImage, CoreImageProps } from '../../core-image';
import { CoreInteractivePublicProps } from '../../core-interactive';
import { Interactable } from '../../interactable';
import { AlignItems, Display, Layout, Position } from '../../layout';
import { BorderRadius } from '../../styled-layout';
import { SVG, SVGAsset } from '../../svg';
import './styles.scss';

type DropDownMenuItemFigure = CoreImageProps | SVGAsset;

export interface DropDownMenuItemProps extends CoreInteractivePublicProps {
  title: string;
  /* Icon or image to display with the menu item. */
  figure?: DropDownMenuItemFigure;
  /* If `true`, aligns the figure to the right of the menu item. */
  figureRight?: boolean;
  /* If using an image as the figure, explicitly set the aspect ratio. Defaults to `Aspect1x1`. */
  figureAspectRatio?: AspectRatio;
}

export const DropDownMenuItem: React.SFC<DropDownMenuItemProps> = (props) => {
  let figure: JSX.Element | undefined;

  if (props.figure && isImage(props.figure)) {
    figure = (
      <Layout
        className="tw-drop-down-menu-item__image"
        padding={{ left: props.figureRight ? 1 : undefined, right: !props.figureRight ? 1 : undefined }}
      >
        <Aspect ratio={props.figureAspectRatio || AspectRatio.Aspect1x1}>
          <CoreImage {...getImageProps(props.figure)} />
        </Aspect>
      </Layout>
    );
  }

  if (props.figure && isIcon(props.figure)) {
    figure = (
      <Layout
        display={Display.Flex}
        alignItems={AlignItems.Center}
        padding={{ right: !props.figureRight ? 0.5 : undefined, left: props.figureRight ? 0.5 : undefined }}
      >
        <SVG asset={props.figure} />
      </Layout>
    );
  }

  return (
    <Layout position={Position.Relative}>
      <Interactable
        borderRadius={BorderRadius.Medium}
        {...props}
      >
        <Layout
          display={Display.Flex}
          alignItems={AlignItems.Center}
          position={Position.Relative}
          padding={0.5}
        >
          {!props.figureRight && figure}
          <Layout flexGrow={1}>{props.title}</Layout>
          {props.figureRight && figure}
        </Layout>
      </Interactable>
    </Layout>
  );
};

function isImage(arg: DropDownMenuItemFigure): arg is CoreImageProps {
  return (arg as CoreImageProps).src !== undefined && (arg as CoreImageProps).alt !== undefined;
}

function isIcon(arg: DropDownMenuItemFigure): arg is SVGAsset {
  return (arg as CoreImageProps).src === undefined;
}

DropDownMenuItem.displayName = 'DropDownMenuItem';
