import * as React from 'react';
import { AriaRole } from '../../aria/aria';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { Color, CoreText, FontSize, TextType } from '../core-text';
import { Column } from '../grid/column';
import { Grid } from '../grid/grid';
import { AlignItems, Display, FlexDirection, JustifyContent, Layout, Position } from '../layout';
import { ResponsiveWrapper } from '../responsive-wrapper';
import { Background, StyledLayout } from '../styled-layout';
import './styles.scss';

export const testSelectors = {
  title: 'editorial-detail-title',
  subtitle: 'editorial-detail-subtitle',
  wrapper: 'editorial-detail-wrapper',
};

export interface EditorialDetailProps {
  /**
   * Main text
   * @example Get a new item!
   */
  title: string;

  /**
   * Supplemental text to provide more information
   * @example Multiple items delivered monthly!
   */
  subtitle?: string;

  /**
   * Background color for the editorial with
   * @example AccentAlt2
   */
  backgroundColor?: Background;
}

/**
 * Messaging to describe an element on the page
 */
export const EditorialDetail: React.SFC<EditorialDetailProps> = (props) => {
  const backgroundColor = props.backgroundColor || Background.Accent;
  const textColor = backgroundColor && [Background.Alt, Background.Alt2, Background.Base].indexOf(backgroundColor) > 0 ? Color.Base : Color.Overlay;
  return (
    <StyledLayout
      {...getDataProps(props)}
      role={AriaRole.Contentinfo}
      className={cn('sm-editorial-detail')}
      fontSize={FontSize.Size6}
      color={textColor}
      background={props.backgroundColor || Background.Accent}
      alignItems={AlignItems.Center}
      position={Position.Relative}
      display={Display.Flex}
      fullHeight
      fullWidth
      padding={4}
      data-test-target={testSelectors.wrapper}
    >
      <ResponsiveWrapper centered>
        <Grid justifyContent={JustifyContent.Center}>
          <Column cols={{ default: 12, lg: 9, xl: 8, xxl: 7 }}>
            <StyledLayout
              display={Display.Flex}
              alignItems={AlignItems.Center}
              position={Position.Relative}
            >
              <Layout display={Display.Flex} alignItems={AlignItems.Start} flexGrow={1} flexDirection={FlexDirection.Column}>
                  <Layout display={Display.Block}>
                    <CoreText data-test-target={testSelectors.title} type={TextType.H2}>{props.title}</CoreText>
                  </Layout>
                  {(props.subtitle) && <Layout display={Display.Block} margin={{ top: 3 }}>
                    <CoreText data-test-target={testSelectors.subtitle} type={TextType.H4}>{props.subtitle}</CoreText>
                  </Layout>}
              </Layout>
            </StyledLayout>
          </Column>
        </Grid>
      </ResponsiveWrapper>
    </StyledLayout>
  );
};

EditorialDetail.displayName = 'EditorialDetail';
