import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import { newUUIDv4 } from '../../../utils/unique-id';
import { FontSize } from '../../core-text';
import { Display, FlexWrap, Layout, Position } from '../../layout';
import { InjectStyledLayout } from '../../styled-layout';
import { FormError } from '../error';
import { FormHint } from '../hint';
import { FormLabel } from '../label';
import { FormLegend } from '../legend';
import './styles.scss';

export enum FormGroupOrientation {
  Vertical = 1,
  Horizontal,
}

export enum FormGroupControls {
  Single = 1,
  Multiple,
}

export interface FormGroupProps {
  controls?: FormGroupControls;
  /**
   * @example <Input type={InputType.Text} placeholder="Video Title" />
   */
  children?: React.ReactNode;
  error?: boolean;
  errorMessage?: string;
  hint?: string;
  id?: string;
  /**
   * @example Title
   */
  label: string;
  labelOptional?: string;
  required?: boolean;
  orientation?: FormGroupOrientation;
}

export const FormGroup: React.SFC<FormGroupProps> = (props) => {
  let label: JSX.Element | undefined;
  let optionalLabel: JSX.Element | undefined;
  let formHint: JSX.Element | undefined;
  let controls: JSX.Element | undefined;
  let container: 'fieldset' | 'div';

  const generatedId = newUUIDv4();

  const classesFormGroup: ClassValue = {
    'tw-form-group': true,
    'tw-form-group--error': props.error,
  };

  if (props.labelOptional) {
    optionalLabel = (
      <span className="tw-form-group__optional">({props.labelOptional})</span>
    );
  }

  if (props.hint && !props.error) {
    formHint = <FormHint hint={props.hint}/>;
  } else if (props.errorMessage && props.error) {
    formHint = <FormError errorMessage={props.errorMessage}/>;
  }

  if (formHint) {
    formHint = <Layout margin={{ top: 0.5 }}>{formHint}</Layout>;
  }

  if (props.controls === FormGroupControls.Multiple) {
    label = <FormLegend label={props.label}/>;
    container = 'fieldset';
  } else {
    label = <FormLabel id={props.id ? props.id : generatedId} required={props.required} label={props.label} />;
    container = 'div';
  }

  label = <Layout margin={{ bottom: 0.5 }}>{label}{optionalLabel}</Layout>;

  if (props.orientation === FormGroupOrientation.Horizontal) {
    controls = React.createElement(container, null, (
        <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap}>
          <Layout className="tw-form-group__label-container" flexShrink={0} padding={{ right: 2 }}>
            {label}
          </Layout>
          <Layout flexGrow={1}>
            {props.children}
            {formHint}
          </Layout>
        </Layout>
      ),
    );
  } else {
    controls = React.createElement(container, null, (
        <div>
          {label}
          {props.children}
          {formHint}
        </div>
      ),
    );
  }

  return (
    <InjectStyledLayout
      className={cn(classesFormGroup)}
      {...getDataProps(props)}
      position={Position.Relative}
      flexGrow={1}
      fontSize={FontSize.Size6}
    >
      {controls}
    </InjectStyledLayout>
  );
};

FormGroup.displayName = 'FormGroup';
