import * as React from 'react';
import { cn } from '../../../utils/classnames';
import {
  Interactable,
  InteractableType,
} from '../../interactable';
import { AlignItems, Display, JustifyContent, Layout } from '../../layout';
import { BorderRadius, StyledLayout } from '../../styled-layout';
import {
  SVG,
  SVGAsset,
} from '../../svg';
import { INPUT_SIZES, InputSize } from '../form';
import './styles.scss';

export interface FormTagProps {
  /** Removes handlers and displays the tag as disabled. */
  disabled?: boolean;
  /** Text to display within the tag. */
  label: string;
  /** Event called when the tag is clicked. */
  onClick?: React.MouseEventHandler<HTMLElement>;
  /** Size of the tag. */
  size?: InputSize;
  /** Displays the tag as "selected". */
  selected?: boolean;
  /** Manually set the `tabIndex` of the tag. */
  tabIndex?: number;
}

export const FormTag: React.SFC<FormTagProps> = (props) => {

  const classes: ClassValue = {
    'tw-form-tag': true,
    'tw-form-tag--selected': props.selected,
  };

  let inputSize = InputSize.Default;
  if (props.size) {
    inputSize = props.size;
  }

  let tabIndex: number | undefined;
  if (props.tabIndex) {
    tabIndex = props.tabIndex;
  }

  return (
    <StyledLayout
      display={Display.InlineFlex}
      alignItems={AlignItems.Center}
      borderRadius={BorderRadius.Medium}
      fontSize={INPUT_SIZES[inputSize]}
      className={cn(classes)}
    >
      <Interactable
        type={InteractableType.Alt}
        borderRadius={BorderRadius.Medium}
        onClick={props.onClick}
        disabled={props.disabled}
        tabIndex={tabIndex}
      >
        <Layout
          display={Display.Flex}
          alignItems={AlignItems.Center}
          padding={{ left: 1, y: 0.5, right: 0.5 }}
        >
          {props.label}

          <StyledLayout
            display={Display.Flex}
            alignItems={AlignItems.Center}
            justifyContent={JustifyContent.Center}
            borderRadius={BorderRadius.Rounded}
            padding={{ left: 0.5 }}
          >
            <SVG asset={props.selected ? SVGAsset.RemoveTag : SVGAsset.Plus} height={10} />
          </StyledLayout>
        </Layout>
      </Interactable>
    </StyledLayout>
  );
};

FormTag.displayName = 'FormTag';
