import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import { Color } from '../../core-text';
import { AlignItems, Display, JustifyContent, Layout, Position, ZIndex } from '../../layout';
import { BorderRadius, InjectStyledLayout, StyledLayout } from '../../styled-layout';
import { SVG, SVGAsset, SVGType } from '../../svg';
import { INPUT_SIZES, InputSize } from '../form';
import './styles.scss';

export enum InputType {
  Text,
  Number,
  Email,
  Password,
  Search,
}

export interface InputProps {
  ariaLabel?: string;
  autoCapitalize?: boolean;
  autoComplete?: string;
  autoCorrect?: boolean;
  autoFocus?: boolean;
  disabled?: boolean;
  error?: boolean;
  icon?: SVGAsset;
  iconRight?: boolean;
  id?: string;
  list?: string;
  maxLength?: number;
  name?: string;
  onBlur?: React.FormEventHandler<HTMLInputElement>;
  onChange?: React.FormEventHandler<HTMLInputElement>;
  onClick?: React.MouseEventHandler<HTMLElement>;
  onFocus?: React.FormEventHandler<HTMLInputElement>;
  onKeyDown?: React.KeyboardEventHandler<HTMLInputElement>;
  placeholder?: string;
  readOnly?: boolean;
  required?: boolean;
  spellCheck?: boolean;
  tabIndex?: number;
  min?: number;
  max?: number;
  size?: InputSize;
  step?: number;
  /**
   * @example Text
   */
  type: InputType;
  value?: string;
  refDelegate?: (e: HTMLInputElement) => void;
  defaultValue?: string;
}

export const Input: React.SFC<InputProps> = (props) => {
  let icon: JSX.Element | undefined;

  let tabIndex: number | undefined;
  if (props.tabIndex) {
    tabIndex = props.tabIndex;
  }

  let inputSize = InputSize.Default;
  if (props.size) {
    inputSize = props.size;
  }

  const classes: ClassValue = {
    'tw-input': true,
    'tw-input--error': props.error,
  };

  if (props.icon) {
    icon = (
      <StyledLayout
        className="tw-input__icon"
        position={Position.Absolute}
        attachLeft={props.icon && !props.iconRight}
        attachRight={props.iconRight}
        attachTop
        zIndex={ZIndex.Default}
        color={Color.Alt2}
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Center}
        fullHeight
      >
        <SVG asset={props.icon} type={SVGType.Inherit} />
      </StyledLayout>
    );
  }

  return (
    <Layout position={Position.Relative} {...getDataProps(props)}>
      {icon}
      <InjectStyledLayout
        display={Display.Block}
        borderRadius={BorderRadius.Medium}
        padding={{
          left: (props.icon && !props.iconRight ? 3 : 1),
          right: (props.iconRight ? 3 : 1),
          y: 0.5,
        }}
        fontSize={INPUT_SIZES[inputSize]}
        fullWidth
      >
        <input
          type={InputType[props.type].toLowerCase()}
          className={cn(classes)}
          placeholder={props.placeholder}
          aria-label={props.ariaLabel}
          autoCapitalize={props.autoCapitalize ? 'on' : 'off'}
          autoCorrect={props.autoCorrect ? 'on' : 'off'}
          autoComplete={props.autoComplete}
          autoFocus={props.autoFocus}
          data-a-target="tw-input"
          defaultValue={props.defaultValue}
          disabled={props.disabled}
          id={props.id}
          list={props.list}
          name={props.name}
          onBlur={props.onBlur}
          onChange={props.onChange}
          onClick={props.onClick}
          onFocus={props.onFocus}
          onKeyDown={props.onKeyDown}
          max={props.max}
          maxLength={props.maxLength}
          min={props.min}
          readOnly={props.readOnly}
          ref={props.refDelegate}
          required={props.required}
          spellCheck={props.spellCheck}
          step={props.step}
          tabIndex={tabIndex}
          value={props.value}
        />
      </InjectStyledLayout>
    </Layout>
  );
};

Input.displayName = 'Input';
