import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import { Display, Overflow } from '../../layout';
import { BorderRadius, InjectStyledLayout } from '../../styled-layout';
import { INPUT_SIZES, InputSize } from '../form';
import './styles.scss';

export interface TextAreaProps {
  autoFocus?: boolean;
  cols?: number;
  disabled?: boolean;
  error?: boolean;
  id?: string;
  maxLength?: number;
  name?: string;
  onBlur?: React.FormEventHandler<HTMLTextAreaElement>;
  onChange?: React.FormEventHandler<HTMLTextAreaElement>;
  onFocus?: React.FormEventHandler<HTMLTextAreaElement>;
  onKeyDown?: React.KeyboardEventHandler<HTMLTextAreaElement>;
  overflow?: Overflow;
  paddingRight?: number;
  placeholder?: string;
  noResize?: boolean;
  readOnly?: boolean;
  rows?: number;
  size?: InputSize;
  tabIndex?: number;
  value?: string;
  refDelegate?: (e: HTMLTextAreaElement) => void;
  defaultValue?: string;
}

export const TextArea: React.SFC<TextAreaProps> = (props) => {
  const classesTextarea: ClassValue = {
    'tw-textarea': true,
    'tw-textarea--no-resize':  props.noResize,
    'tw-textarea--error':  props.error,
  };

  let tabIndex: number | undefined;
  if (props.tabIndex) {
    tabIndex = props.tabIndex;
  }

  let inputSize = InputSize.Default;
  if (props.size) {
    inputSize = props.size;
  }

  const paddingRight: React.CSSProperties = { paddingRight: props.paddingRight ? props.paddingRight / 10 + 'rem' : undefined };

  return (
    <InjectStyledLayout
      display={Display.Block}
      borderRadius={BorderRadius.Medium}
      overflow={props.overflow}
      fontSize={INPUT_SIZES[inputSize]}
      fullWidth
    >
      <textarea
        className={cn(classesTextarea)}
        autoFocus={props.autoFocus}
        cols={props.cols}
        disabled={props.disabled}
        id={props.id}
        maxLength={props.maxLength}
        name={props.name}
        onBlur={props.onBlur}
        onChange={props.onChange}
        onFocus={props.onFocus}
        onKeyDown={props.onKeyDown}
        style={paddingRight}
        placeholder={props.placeholder}
        rows={props.rows}
        tabIndex={tabIndex}
        value={props.value}
        defaultValue={props.defaultValue}
        ref={props.refDelegate}
        readOnly={props.readOnly}
        {...getDataProps(props)}
      />
    </InjectStyledLayout>
  );
};

TextArea.displayName = 'TextArea';
