import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import './styles.scss';

export type Columns = 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10 | 11 | 12;
export type OffsetColumns = 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10 | 11;

export enum ColumnOrder {
  First = 'col-order--first',
  Last = 'col-order--last',
}

export interface ColumnValues {
  // The default key will always be required when sending a breakpoint object,
  // as it represents all viewport sizes above zero.
  default: Columns;
  xs?: Columns;
  sm?: Columns;
  md?: Columns;
  lg?: Columns;
  xl?: Columns;
  xxl?: Columns;
}

export interface OffsetValues {
  // The default key will always be required when sending a breakpoint object, as it represents all viewport sizes above zero.
  default: OffsetColumns;
  xs?: OffsetColumns;
  sm?: OffsetColumns;
  md?: OffsetColumns;
  lg?: OffsetColumns;
  xl?: OffsetColumns;
  xxl?: OffsetColumns;
}

export interface ColumnOrderValues {
  default?: ColumnOrder;
  xs?: ColumnOrder;
  sm?: ColumnOrder;
  md?: ColumnOrder;
  lg?: ColumnOrder;
  xl?: ColumnOrder;
  xxl?: ColumnOrder;
}

export interface ColumnProps {
  children?: React.ReactNode;
  /**
   * @example 4
   */
  cols: ColumnValues | Columns;
  offset?: OffsetValues | OffsetColumns;
  order?: ColumnOrderValues | ColumnOrder;
}

export class Column extends React.Component<ColumnProps, {}> {
  public render() {
    const columnSizeClasses = this.getColumnSizeClasses(this.props.cols);
    const columnOffsetClasses = this.props.offset ? this.getColumnOffsetClasses(this.props.offset) : '';
    const columnOrderClasses = this.props.order ? this.getColumnOrderClasses(this.props.order) : '';

    return (
      <div className={cn([columnSizeClasses, columnOffsetClasses, columnOrderClasses])} {...getDataProps(this.props)}>
        {this.props.children}
      </div>
    );
  }

  private getColumnSizeClasses(value: Columns | ColumnValues) {
    if (typeof value === 'number') {
      return `tw-col-${value}`;
    }

    const classes: string[] = [`tw-col-${value.default}`];

    if (typeof value.xs === 'number') {
      classes.push(`tw-xs-col-${value.xs}`);
    }

    if (typeof value.sm === 'number') {
      classes.push(`tw-sm-col-${value.sm}`);
    }

    if (typeof value.md === 'number') {
      classes.push(`tw-md-col-${value.md}`);
    }

    if (typeof value.lg === 'number') {
      classes.push(`tw-lg-col-${value.lg}`);
    }

    if (typeof value.xl === 'number') {
      classes.push(`tw-xl-col-${value.xl}`);
    }

    if (typeof value.xxl === 'number') {
      classes.push(`tw-xxl-col-${value.xxl}`);
    }

    return classes;
  }

  private getColumnOffsetClasses(value: OffsetColumns | OffsetValues) {
    if (typeof value === 'number') {
      return `tw-col-offset-${value}`;
    }

    const classes: string[] = [`tw-col-offset-${value.default}`];

    if (typeof value.xs === 'number') {
      classes.push(`tw-xs-col-offset-${value.xs}`);
    }

    if (typeof value.sm === 'number') {
      classes.push(`tw-sm-col-offset-${value.sm}`);
    }

    if (typeof value.md === 'number') {
      classes.push(`tw-md-col-offset-${value.md}`);
    }

    if (typeof value.lg === 'number') {
      classes.push(`tw-lg-col-offset-${value.lg}`);
    }

    if (typeof value.xl === 'number') {
      classes.push(`tw-xl-col-offset-${value.xl}`);
    }

    if (typeof value.xxl === 'number') {
      classes.push(`tw-xxl-col-offset-${value.xxl}`);
    }

    return classes;
  }

  private getColumnOrderClasses(value: ColumnOrder | ColumnOrderValues) {
    if (typeof value === 'string') {
      return `tw-${value}`;
    } else if (typeof value === 'object') {

      const classes: string[] = [];

      if (typeof value.default === 'string') {
        classes.push(`tw-${value.default}`);
      }

      if (typeof value.xs === 'string') {
        classes.push(`tw-xs-${value.xs}`);
      }

      if (typeof value.sm === 'string') {
        classes.push(`tw-sm-${value.sm}`);
      }

      if (typeof value.md === 'string') {
        classes.push(`tw-md-${value.md}`);
      }

      if (typeof value.lg === 'string') {
        classes.push(`tw-lg-${value.lg}`);
      }

      if (typeof value.xl === 'string') {
        classes.push(`tw-xl-${value.xl}`);
      }

      if (typeof value.xxl === 'string') {
        classes.push(`tw-xxl-${value.xxl}`);
      }

      return classes;
    } else {
      return false;
    }
  }
}
