import * as React from 'react';
import { CoreImage, StyledLayout } from '../..';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { AlignItems, Display, Layout, Position, ZIndex } from '../layout';
import './styles.scss';

export enum HeroChildrenPosition {
  LeftTop = 'left-top',
  LeftBottom = 'left-bottom',
  RightTop = 'right-top',
  RightBottom = 'right-bottom',
}

export const testSelectors = {
  wrapper: 'hero-wrapper',
};

export interface HeroProps {
  /**
   * Path to hero image
   * @example https://m.media-amazon.com/images/G/01//mchenryz/ExportedImage._CB1198675309_.png
   */
  imageSrc: string;

  /**
   * Child component
   */
  children?: React.ReactNode;

  /**
   * Position of the children with respect to the parent position
   */
  childrenPosition?: HeroChildrenPosition;

}

/**
 * Hero to be used for epic stuff
 */
export const Hero: React.SFC<HeroProps> = (props) => {
  return (
    <StyledLayout
      {...getDataProps(props)}
      className={cn('sm-hero')}
      alignItems={AlignItems.Center}
      position={Position.Relative}
      display={Display.Flex}
      fullHeight
      fullWidth
      data-test-target={testSelectors.wrapper}
    >
        <Layout zIndex={ZIndex.Default} fullHeight fullWidth>
          <CoreImage src={props.imageSrc} alt={'Content Image'} fullHeight fullWidth />
        </Layout>
        {props.children && props.childrenPosition && renderChildren(props.children, props.childrenPosition)}
    </StyledLayout>
  );
};

const renderChildren = (children: React.ReactNode, childrenPosition: HeroChildrenPosition) => {
  const childHorizontalPosition = childrenPosition.split('-')[0];
  const childVerticalPosition = childrenPosition.split('-')[1];
  return (
    <Layout
      position={Position.Absolute}
      attachTop={childVerticalPosition === 'top'}
      attachBottom={childVerticalPosition === 'bottom'}
      attachLeft={childHorizontalPosition === 'left'}
      attachRight={childHorizontalPosition === 'right'}
      zIndex={ZIndex.Above}
    >
      {children}
    </Layout>
  );
};

Hero.displayName = 'Hero';
