import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { Avatar, AvatarProps } from '../avatar';
import { Button, ButtonProps, ButtonSize, ButtonType } from '../button/button';
import { ButtonIcon, ButtonIconSize, ButtonIconType } from '../button/button-icon';
import { CoreLink, CoreLinkProps } from '../core-link';
import { CoreText, FontSize, TextType } from '../core-text';
import { AlignItems, Display, FlexDirection, Layout, Position } from '../layout';
import { Background, BorderRadius, StyledLayout } from '../styled-layout';
import { SVG, SVGAsset } from '../svg';
import './styles.scss';

export interface InFeatureNotificationProps {
  /**
   * Up to two actionable buttons as part of the banner.
   */
  actions?: [ButtonProps, ButtonProps | undefined];
  /**
   * Translated string for Close button (x) ariaLabel.
   */
  closeButtonAriaLabel: string;
  /**
   * Optional avatar to replace the default SVG icon.
   */
  avatar?: AvatarProps;
  /**
   * Specify a background color for the notification.
   */
  background?: Background.Base | Background.Alt | Background.Alt2;
  /**
   * Long text for the description.
   */
  description: string;
  /**
   * Optional link CTA for further context.
   */
  link?: CoreLinkProps;
  /**
   * Callback when the close button is clicked.
   */
  closeButtonOnClick?: () => void;
  /**
   * Narrow notification.
   */
  narrow?: boolean;
  /**
   * Short text for the title.
   */
  title: string;
  /**
   * The type of in-feature notification.
   * @example Error
   */
  type: InFeatureNotificationType;
}

export enum InFeatureNotificationType {
  Error = 'tw-in-feature-notification--error',
  Info = 'tw-in-feature-notification--info',
  Neutral = 'tw-in-feature-notification--neutral',
  Success = 'tw-in-feature-notification--success',
  Warning = 'tw-in-feature-notification--warning',
}

const IN_FEATURE_NOTIFICATION_ICON: { [key: string]: SVGAsset } = {
  [InFeatureNotificationType.Error]: SVGAsset.NotificationError,
  [InFeatureNotificationType.Info]: SVGAsset.NotificationInfo,
  [InFeatureNotificationType.Success]: SVGAsset.NotificationSuccess,
  [InFeatureNotificationType.Warning]: SVGAsset.NotificationWarning,
};

export const InFeatureNotification: React.SFC<InFeatureNotificationProps> = (props) => {
  const classes: ClassValue = {
    'tw-in-feature-notification': true,
    [props.type]: true,
  };

  let actions: JSX.Element | null = null;
  let icon: JSX.Element | null = null;
  let link = props.link ? <CoreLink {...props.link} /> : null;

  const closeButton = (
    <Layout
      attachRight
      attachTop
      margin={{ right: 0.5, top: 0.5 }}
      position={Position.Absolute}
    >
      <ButtonIcon
        ariaLabel={props.closeButtonAriaLabel}
        icon={SVGAsset.Close}
        onClick={props.closeButtonOnClick}
        size={ButtonIconSize.Small}
        type={ButtonIconType.Secondary}
      />
    </Layout>
  );

  if (props.avatar && !props.narrow) {
    icon = (
      <Layout
        alignItems={AlignItems.Start}
        display={Display.Flex}
        margin={{ right: 1 }}
      >
        <Avatar {...props.avatar} size={30} />
      </Layout>
    );
  } else if (props.type !== InFeatureNotificationType.Neutral) {
    icon = (
      <Layout
        className="tw-in-feature-notification__icon"
        alignItems={AlignItems.Center}
        display={Display.Flex}
        margin={{ right: 1 }}
      >
        <SVG asset={IN_FEATURE_NOTIFICATION_ICON[props.type]} />
      </Layout>
    );
  }

  // Show CTA's if ButtonProps are passed in to the actions prop
  // and narrow mode is NOT enabled.
  // The `&& !props.narrow` may be removed to show CTA's when
  // narrow mode IS enabled.
  if (props.actions && !props.narrow) {
    actions = (
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.RowReverse}
        fullWidth
        margin={{ top: 1 }}
      >
        {props.actions.map((action, i) => (
          <Layout key={`in-feature-notification-${i}`} margin={{ left: 1 }}>
            <Button {...action} size={ButtonSize.Small} type={i === 0 ? ButtonType.Default : ButtonType.Hollow} />
          </Layout>),
        )}
      </Layout>
    );
  }

  const content = props.narrow ? (
    <CoreText fontSize={FontSize.Size6}>
      <CoreText bold type={TextType.Span}>{props.title}</CoreText> {props.description} {link}
    </CoreText>
  ) : (
    <>
      <CoreText bold fontSize={FontSize.Size6}>
        {props.title}
      </CoreText>
      <CoreText fontSize={FontSize.Size6}>
        {props.description}
      </CoreText>
      {link}
    </>
  );

  return (
    <StyledLayout
      {...getDataProps(props)}
      className={cn(classes)}
      background={props.background || Background.Base}
      borderRadius={BorderRadius.Medium}
      padding={1}
      position={Position.Relative}
    >
      {closeButton}
      <Layout display={Display.Flex} fullWidth>
        {icon}
        <Layout fullWidth padding={{ right: 2 }}>
          {content}
        </Layout>
      </Layout>
      {actions}
    </StyledLayout>
  );
};

InFeatureNotification.displayName = 'InFeatureNotification';
