import * as React from 'react';
import { AriaProps } from '../../aria/aria';
import { getAriaProps } from '../../utils/aria-props';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import './styles.scss';

export enum AlignContent {
  Start = 'align-content-start',
  End = 'align-content-end',
  Center = 'align-content-center',
  Baseline = 'align-content-baseline',
  Stretch = 'align-content-stretch',
}

export enum AlignItems {
  Start = 'align-items-start',
  End = 'align-items-end',
  Center = 'align-items-center',
  Baseline = 'align-items-baseline',
  Stretch = 'align-items-stretch',
}

export enum AlignSelf {
  Start = 'align-self-start',
  End = 'align-self-end',
  Center = 'align-self-center',
  Baseline = 'align-self-baseline',
  Stretch = 'align-self-stretch',
}

export enum Display {
  Block = 'block',
  Flex = 'flex',
  Inline = 'inline',
  InlineBlock = 'inline-block',
  InlineFlex = 'inline-flex',
  Hide = 'hide',
  HideAccessible = 'hide-accessible',
}

export enum FlexDirection {
  Column = 'flex-column',
  ColumnReverse = 'flex-column-reverse',
  Row = 'flex-row',
  RowReverse = 'flex-row-reverse',
}

export enum FlexWrap {
  Wrap = 'flex-wrap',
  NoWrap = 'flex-nowrap',
  WrapReverse = 'flex-wrap-reverse',
}

export enum JustifyContent {
  Start = 'justify-content-start',
  End = 'justify-content-end',
  Center = 'justify-content-center',
  Between = 'justify-content-between',
  Around = 'justify-content-around',
}

export enum Overflow {
  Auto = 'tw-overflow-auto',
  Scroll = 'tw-overflow-scroll',
  Visible = 'tw-overflow-visible',
  Hidden = 'tw-overflow-hidden',
}

export enum Position {
  Relative = 'relative',
  Absolute = 'absolute',
  Fixed = 'fixed',
}

export enum Resize {
  None = 'tw-resize-none',
  X = 'tw-resize-x',
  Y = 'tw-resize-y',
}

export enum TextAlign {
  Left = 'align-left',
  Center = 'align-center',
  Right = 'align-right',
  Justify = 'align-justify',
}

export enum VerticalAlign {
  Top = 'tw-align-top',
  Middle = 'tw-align-middle',
  Baseline = 'tw-align-baseline',
  Bottom = 'tw-align-bottom',
  TextTop = 'tw-align-text-top',
  TextBottom = 'tw-align-text-bottom',
}

export enum Visibility {
  Visible = 'visible',
  Hidden = 'hidden',
}

export enum ZIndex {
  Default = 'tw-z-default',
  Above = 'tw-z-above',
  Below = 'tw-z-below',
}

export type FlexValue = 0 | 1 | 2 | 3 | 4;
export type PaddingValue = 0 | 0.5 | 1 | 2 | 3 | 4 | 5;
export type MarginValue = PaddingValue | 'auto';

export interface PaddingValues {
  top?: PaddingValue;
  right?: PaddingValue;
  bottom?: PaddingValue;
  left?: PaddingValue;
  x?: PaddingValue;
  y?: PaddingValue;
}

export interface MarginValues {
  top?: MarginValue;
  right?: MarginValue;
  bottom?: MarginValue;
  left?: MarginValue;
  x?: MarginValue;
  y?: MarginValue;
}

export type Padding = PaddingValue | PaddingValues;
export type Margin = MarginValue | MarginValues;

const VALID_VALUES = new Set([0, 0.5, 1, 2, 3, 4, 5, 'auto']);

export interface BreakpointProps {
  alignContent?: AlignContent;
  alignItems?: AlignItems;
  alignSelf?: AlignSelf;
  display?: Display;
  flexDirection?: FlexDirection;
  flexGrow?: FlexValue;
  flexOrder?: FlexValue;
  flexShrink?: FlexValue;
  flexWrap?: FlexWrap;
  justifyContent?: JustifyContent;
  margin?: Margin;
  padding?: Padding;
  position?: Position;
  textAlign?: TextAlign;
  visibility?: Visibility;
}

export interface InjectLayoutProps extends BreakpointProps, AriaProps {
  attachTop?: boolean;
  attachRight?: boolean;
  attachBottom?: boolean;
  attachLeft?: boolean;
  children?: React.ReactNode;
  className?: string;
  ellipsis?: boolean;
  fullHeight?: boolean;
  fullWidth?: boolean;
  overflow?: Overflow;
  resize?: Resize;
  verticalAlign?: VerticalAlign;
  zIndex?: ZIndex;
  breakpointExtraSmall?: BreakpointProps;
  breakpointSmall?: BreakpointProps;
  breakpointMedium?: BreakpointProps;
  breakpointLarge?: BreakpointProps;
  breakpointExtraLarge?: BreakpointProps;
  breakpointExtraExtraLarge?: BreakpointProps;
}

export class InjectLayout extends React.Component<InjectLayoutProps> {
  public render() {
    const classes: ClassValue = {
      [`${this.props.className}`]: !!this.props.className,
      'tw-top-0': this.props.attachTop,
      'tw-right-0': this.props.attachRight,
      'tw-bottom-0': this.props.attachBottom,
      'tw-left-0': this.props.attachLeft,
      'tw-ellipsis': this.props.ellipsis,
      'tw-full-width': this.props.fullWidth,
      'tw-full-height': this.props.fullHeight,
    };

    if (this.props.overflow) {
      classes[this.props.overflow] = true;
    }

    if (this.props.resize) {
      classes[Resize.X] = true;
    }

    if (this.props.verticalAlign) {
      classes[this.props.verticalAlign] = true;
    }

    if (this.props.zIndex) {
      classes[this.props.zIndex] = true;
    }

    const defaultBreakpoint = this.getBreakpointClasses(this.props);
    const extraSmallBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraSmall, 'xs');
    const smallBreakpoint = this.getBreakpointClasses(this.props.breakpointSmall, 'sm');
    const mediumBreakpoint = this.getBreakpointClasses(this.props.breakpointMedium, 'md');
    const largeBreakpoint = this.getBreakpointClasses(this.props.breakpointLarge, 'lg');
    const extraLargeBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraLarge, 'xl');
    const extraExtraLargeBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraExtraLarge, 'xxl');

    const child = React.Children.only(this.props.children);
    const newChild = React.cloneElement(child, {
      ...child.props,
      ...getDataProps(this.props),
      ...getAriaProps(this.props),
      className: cn(
        child.props.className,
        classes,
        defaultBreakpoint,
        extraSmallBreakpoint,
        smallBreakpoint,
        mediumBreakpoint,
        largeBreakpoint,
        extraLargeBreakpoint,
        extraExtraLargeBreakpoint,
      ),
    });

    return newChild;
  }

  private getBreakpointClasses(value?: BreakpointProps, prefix?: string) {
    if (!value) {
      return '';
    }

    let classes: string[] = [];

    if (prefix) {
      prefix = 'tw-' + prefix + '-';
    } else {
      prefix = 'tw-';
    }

    if (value.alignContent) {
      classes.push(prefix + value.alignContent);
    }

    if (value.alignItems) {
      classes.push(prefix + value.alignItems);
    }

    if (value.alignSelf) {
      classes.push(prefix + value.alignSelf);
    }

    if (value.display) {
      classes.push(prefix + value.display);
    }

    if (value.flexDirection) {
      classes.push(prefix + value.flexDirection);
    }

    if (value.flexGrow !== undefined) {
      classes.push(`${prefix}flex-grow-${value.flexGrow}`);
    }

    if (value.flexOrder !== undefined) {
      classes.push(`${prefix}item-order-${value.flexOrder}`);
    }

    if (value.flexShrink !== undefined) {
      classes.push(`${prefix}flex-shrink-${value.flexShrink}`);
    }

    if (value.flexWrap) {
      classes.push(prefix + value.flexWrap);
    }

    if (value.justifyContent) {
      classes.push(prefix + value.justifyContent);
    }

    if (value.position) {
      classes.push(prefix + value.position);
    }

    if (value.textAlign) {
      classes.push(prefix + value.textAlign);
    }

    if (value.visibility) {
      classes.push(prefix + value.visibility);
    }

    if (value.margin !== undefined) {
      const prefixedClasses = this.getSpacingClasses('mg', value.margin);
      if (typeof prefixedClasses === 'string') {
        classes.push(prefix + prefixedClasses);
      } else {
        prefixedClasses.forEach((marginClass) => {
          classes.push(prefix + marginClass);
        });
      }
    }

    if (value.padding !== undefined) {
      const prefixedClasses = this.getSpacingClasses('pd', value.padding);
      if (typeof prefixedClasses === 'string') {
        classes.push(prefix + prefixedClasses);
      } else {
        prefixedClasses.forEach((paddingClass) => {
          classes.push(prefix + paddingClass);
        });
      }
    }

    return classes;
  }

  private getSpacingClasses(prefix: string, value?: Padding | Margin) {
    if (value === undefined || value === null) {
      return '';
    }

    if (typeof value === 'object') {
      if (value.x && (value.left || value.right)) {
        throw new Error('Cannot use `x` and `left` or `right` at the same time.');
      } else if (value.y && (value.top || value.bottom)) {
        throw new Error('Cannot use `y` and `top` or `bottom` at the same time.');
      }

      let spacingClasses: string[] = [];

      if (value.top !== undefined) {
        if (value.top === 0.5) {
          spacingClasses.push(`${prefix}-t-05`);
        } else {
          spacingClasses.push(`${prefix}-t-${value.top}`);
        }
      }

      if (value.right !== undefined) {
        if (value.right === 0.5) {
          spacingClasses.push(`${prefix}-r-05`);
        } else {
          spacingClasses.push(`${prefix}-r-${value.right}`);
        }
      }

      if (value.bottom !== undefined) {
        if (value.bottom === 0.5) {
          spacingClasses.push(`${prefix}-b-05`);
        } else {
          spacingClasses.push(`${prefix}-b-${value.bottom}`);
        }
      }

      if (value.left !== undefined) {
        if (value.left === 0.5) {
          spacingClasses.push(`${prefix}-l-05`);
        } else {
          spacingClasses.push(`${prefix}-l-${value.left}`);
        }
      }

      if (value.x !== undefined) {
        if (value.x === 0.5) {
          spacingClasses.push(`${prefix}-x-05`);
        } else {
          spacingClasses.push(`${prefix}-x-${value.x}`);
        }
      }

      if (value.y !== undefined) {
        if (value.y === 0.5) {
          spacingClasses.push(`${prefix}-y-05`);
        } else {
          spacingClasses.push(`${prefix}-y-${value.y}`);
        }
      }

      return spacingClasses;

    } else {
      if (!VALID_VALUES.has(value) || (prefix === 'tw-pd' && value === 'auto')) {
        throw new Error(`${value} is not a valid property of either margin or padding`);
      }

      if (value === 0.5) {
        return `${prefix}-05`;
      }

      if (value === 'auto') {
        return `${prefix}-auto`;
      }

      return `${prefix}-${value}`;
    }
  }
}

export interface LayoutProps extends InjectLayoutProps {
  refDelegate?: (e: HTMLDivElement) => void;
}

// tslint:disable-next-line:max-classes-per-file
export class Layout extends React.Component<LayoutProps> {
  public render() {
    return (
      <InjectLayout {...this.props}>
        <div ref={this.props.refDelegate}>
          {this.props.children}
        </div>
      </InjectLayout>
    );
  }
}
