import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { AlignItems, Display, JustifyContent, Layout } from '../layout';
import './styles.scss';

export enum SpinnerSize {
  Default = '',
  Small = 'tw-loading-spinner--small',
  Large = 'tw-loading-spinner--large',
}

export interface LoadingSpinnerProps {
  size?: SpinnerSize;
  delay?: number; // By default this is 300ms
  inheritColor?: boolean;
  fillContent?: boolean;
}

export interface State {
  showSpinner: boolean;
}

export class LoadingSpinner extends React.Component<LoadingSpinnerProps, State> {
  public state = {
    showSpinner: this.props.delay === 0,
  };

  private delayTimeout: number | undefined;

  public componentDidMount() {
    const delay = this.props.delay !== undefined ? this.props.delay : 300;
    if (delay > 0) {
      this.delayTimeout = window.setTimeout(() => {
        this.setState({ showSpinner: true });
      }, delay);
    }
  }

  public componentWillUnmount() {
    if (this.delayTimeout) {
      clearTimeout(this.delayTimeout);
    }
  }

  public render() {
    if (!this.state.showSpinner) {
      return null;
    }

    let spinner: JSX.Element | undefined;

    const classes: ClassValue = {
      'tw-loading-spinner': true,
      'tw-loading-spinner--inherit-color': this.props.inheritColor,
    };

    if (this.props.size) {
      classes[this.props.size] = true;
    }

    spinner = <div className={cn(classes)} {...getDataProps(this.props)} />;

    if (this.props.fillContent) {
      spinner = (
        <Layout display={Display.Flex} justifyContent={JustifyContent.Center} alignItems={AlignItems.Center} fullWidth fullHeight>
          {spinner}
        </Layout>
      );
    }

    return spinner;
  }
}
