import * as React from 'react';
import { AlignItems, Display, Layout } from '../../layout';
import { PaginationControl, PaginationControlType } from '../../pagination/pagination-control';
import { PaginationIndex } from '../../pagination/pagination-index';

// Extend the `React.MouseEVent` interface to accept an item to store
// the destination page of a pagination control. The event object with the
// destination page can then be passed back up to the parent.
export interface PaginationMouseEvent<T> extends React.MouseEvent<T> {
  goToPage: number;
}

export interface PaginationProps {
  /**
   * The currently viewed page.
   * @example 1
   */
  currentPage: number;
  /**
   * The currently viewed page.
   * @example 10
   */
  totalPages: number;
  /**
   * Allow customization and/or translation of the `ariaLabel` on the
   * previous page button.
   *
   * @example Previous Page
   */
  previousPageButtonAriaLabel: string;
  /**
   * Allow customization and/or translation of the `ariaLabel` on the
   * next page button.
   *
   * @example Next Page
   */
  nextPageButtonAriaLabel: string;
  /**
   * Click handler for the "Previous" page control. Returns the number of
   * the page.
   */
  onClickPrevious?: React.MouseEventHandler<HTMLElement>;
  /**
   * Click handler for the "Next" page control. Returns the number of
   * the page.
   */
  onClickNext?: React.MouseEventHandler<HTMLElement>;
  /**
   * Click handler for the page number control. Returns the number of
   * the page. Returns an event object (`PaginationMouseEvent<HTMLDivElement>`).
   */
  onClickIndex?: React.MouseEventHandler<HTMLElement>;
}

export class Pagination extends React.Component<PaginationProps, {}> {

  public render() {

    return (
      <Layout display={Display.Flex} alignItems={AlignItems.Center}>
        <PaginationControl
          ariaLabel={this.props.previousPageButtonAriaLabel}
          type={PaginationControlType.Previous}
          currentPage={this.props.currentPage}
          totalPages={this.props.totalPages}
          onClick={this.handleClickPrevious}
          key="previous"
        />
        <PaginationIndex
          currentPage={this.props.currentPage}
          totalPages={this.props.totalPages}
          onClick={this.props.onClickIndex}
        />
        <PaginationControl
          ariaLabel={this.props.nextPageButtonAriaLabel}
          type={PaginationControlType.Next}
          currentPage={this.props.currentPage}
          totalPages={this.props.totalPages}
          onClick={this.handleClickNext}
          key="next"
        />
      </Layout>
    );
  }

  private handleClickPrevious = (e: PaginationMouseEvent<HTMLElement>) => {
    if (this.props.onClickPrevious) {
      e.goToPage = this.props.currentPage - 1;
      this.props.onClickPrevious(e);
    }
  }

  private handleClickNext = (e: PaginationMouseEvent<HTMLElement>) => {
    if (this.props.onClickNext) {
      e.goToPage = this.props.currentPage + 1;
      this.props.onClickNext(e);
    }
  }

}
