import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { Display } from '../layout';
import { BorderRadius, InjectStyledLayout } from '../styled-layout';
import './styles.scss';

export enum ProgressBarSize {
  Default = '',
  Small = 'tw-progress-bar--sm',
  ExtraSmall = 'tw-progress-bar--xs',
}

export enum ProgressBarStatus {
  Default = '',
  Caution = 'tw-progress-bar--caution',
  Error = 'tw-progress-bar--error',
  Success = 'tw-progress-bar--success',
}

export enum ProgressBarAnimationDirection {
  Up = 'up',
  Down = 'down',
}

export interface ProgressBarProps {
  animationDuration?: number;
  animationDirection?: ProgressBarAnimationDirection;
  borderRadius?: BorderRadius;
  onAnimationEnd?: (e: React.AnimationEvent<HTMLDivElement>) => void;
  size?: ProgressBarSize;
  status?: ProgressBarStatus;
  value?: number;
  inherit?: boolean;
  mask?: boolean;
}

export const ProgressBar: React.SFC<ProgressBarProps> = (props) => {
  const classes: ClassValue = {
    'tw-progress-bar': true,
    'tw-progress-bar--countdown': props.animationDirection === ProgressBarAnimationDirection.Down,
    'tw-progress-bar--countup': props.animationDirection === ProgressBarAnimationDirection.Up,
    'tw-progress-bar--inherit': props.inherit,
    'tw-progress-bar--mask': props.mask,
  };

  if (props.size) {
    classes[props.size] = true;
  }

  if (props.status) {
    classes[props.status] = true;
  }

  const borderRadius = props.borderRadius || BorderRadius.Large;

  return (
    <InjectStyledLayout borderRadius={borderRadius}>
      <div
        className={cn(classes)}
        {...getDataProps(props)}
        role="progressbar"
        aria-valuenow={props.value}
        aria-valuemin={0}
        aria-valuemax={100}
      >
        <InjectStyledLayout borderRadius={borderRadius} display={Display.Block}>
          <div
            onAnimationEnd={props.onAnimationEnd}
            className="tw-progress-bar__fill"
            style={{
              width: `${props.value}%`,
              animationDuration: props.animationDuration ? `${props.animationDuration}s` : '',
            }}
            data-a-target="tw-progress-bar-animation"
          />
        </InjectStyledLayout>
      </div>
    </InjectStyledLayout>
  );
};

ProgressBar.displayName = 'ProgressBar';
