import * as React from 'react';
import { AriaProps } from '../../aria/aria';
import { getAriaProps } from '../../utils/aria-props';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { Color, FontSize } from '../core-text';
import { BreakpointProps as LayoutBreakpointProps, InjectLayout, LayoutProps } from '../layout';
import './styles.scss';

export enum Background {
  Base = 'tw-c-background-base',
  Alt = 'tw-c-background-alt',
  Alt2 = 'tw-c-background-alt-2',
  Overlay = 'tw-c-background-overlay',
  Accent = 'tw-c-background-accent',
  AccentAlt = 'tw-c-background-accent-alt',
  AccentAlt2 = 'tw-c-background-accent-alt-2',
}

export enum BorderRadius {
  None = 'tw-border-radius-none',
  Small = 'tw-border-radius-small',
  Medium = 'tw-border-radius-medium',
  Large = 'tw-border-radius-large',
  Rounded = 'tw-border-radius-rounded',
}

export type Elevation = 0 | 1 | 2 | 3 | 4 | 5;

export interface BorderRadiusValues {
  topLeft?: BorderRadius;
  topRight?: BorderRadius;
  bottomRight?: BorderRadius;
  bottomLeft?: BorderRadius;
}

export interface BreakpointProps extends LayoutBreakpointProps {
  fontSize?: FontSize;
}

export interface StyledLayoutProps extends LayoutProps, BreakpointProps, AriaProps {
  background?: Background;
  border?: boolean;
  borderTop?: boolean;
  borderRight?: boolean;
  borderBottom?: boolean;
  borderLeft?: boolean;
  borderMarked?: boolean;
  borderRadius?: BorderRadius | BorderRadiusValues;
  children?: React.ReactNode;
  color?: Color;
  elevation?: Elevation;
  breakpointExtraSmall?: BreakpointProps;
  breakpointSmall?: BreakpointProps;
  breakpointMedium?: BreakpointProps;
  breakpointLarge?: BreakpointProps;
  breakpointExtraLarge?: BreakpointProps;
  breakpointExtraExtraLarge?: BreakpointProps;
}

export class InjectStyledLayout extends React.Component<StyledLayoutProps, {}> {
  public render() {
    const classes: ClassValue = {
      'tw-border-t': this.props.border || this.props.borderMarked || this.props.borderTop,
      'tw-border-r': this.props.border || this.props.borderMarked || this.props.borderRight,
      'tw-border-b': this.props.border || this.props.borderMarked || this.props.borderBottom,
      'tw-border-l': this.props.border || this.props.borderLeft && !this.props.borderMarked,
      'tw-border--marked': this.props.borderMarked,
      [`tw-elevation-${this.props.elevation}`]: this.props.elevation !== undefined && this.props.elevation > 0,
    };

    if (this.props.borderRadius) {
      if (typeof this.props.borderRadius === 'string') {
        classes[this.props.borderRadius] = true;
      } else {
        this.getBorderRadiusClasses(this.props.borderRadius).forEach((direction) => {
          classes[direction] = true;
        });
      }
    }

    const colorClass = this.getColorClass();
    const backgroundClass = this.getBackgroundClass();

    const defaultBreakpoint = this.getBreakpointClasses(this.props);
    const extraSmallBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraSmall, 'xs');
    const smallBreakpoint = this.getBreakpointClasses(this.props.breakpointSmall, 'sm');
    const mediumBreakpoint = this.getBreakpointClasses(this.props.breakpointMedium, 'md');
    const largeBreakpoint = this.getBreakpointClasses(this.props.breakpointLarge, 'lg');
    const extraLargeBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraLarge, 'xl');
    const extraExtraLargeBreakpoint = this.getBreakpointClasses(this.props.breakpointExtraExtraLarge, 'xxl');

    const layoutProps = {
      ...this.props,
      ...getDataProps(this.props),
      ...getAriaProps(this.props),
      className: cn(
        this.props.className,
        classes,
        backgroundClass,
        colorClass,
        defaultBreakpoint,
        extraSmallBreakpoint,
        smallBreakpoint,
        mediumBreakpoint,
        largeBreakpoint,
        extraLargeBreakpoint,
        extraExtraLargeBreakpoint,
      ),
    };

    return <InjectLayout {...layoutProps} />;
  }

  private getBorderRadiusClasses(values: BorderRadiusValues) {
    let classes: Array<string> = [];

    Object.keys(values).map((key) => {
      let value: BorderRadius | undefined;

      switch (key) {
        case 'topLeft':
          value = values.topLeft;
          break;

        case 'topRight':
          value = values.topRight;
          break;

        case 'bottomRight':
          value = values.bottomRight;
          break;

        case 'bottomLeft':
          value = values.bottomLeft;
          break;

        default:
          return;
      }

      if (value) {
        const direction = key.replace( /([a-z])([A-Z])/g, '$1-$2' ).toLowerCase();
        classes.push(value.replace(/border-radius/, `border-${direction}-radius`));
      }
    });

    return classes;
  }

  private getBreakpointClasses(value?: StyledLayoutProps, prefix?: string) {
    if (!value) {
      return '';
    }

    let breakpointClasses: string[] = [];

    if (prefix) {
      prefix = 'tw-' + prefix + '-';
    } else {
      prefix = 'tw-';
    }

    if (value.fontSize) {
      breakpointClasses.push(prefix + value.fontSize);
    }

    return breakpointClasses;
  }

  private getColorClass() {
    if (this.props.background === Background.Overlay && !this.props.color) {
      return Color.Overlay;
    } else if (this.props.color) {
      return this.props.color;
    }
    return '';
  }

  private getBackgroundClass() {
    if (this.props.background) {
      return this.props.background;
    }
    return '';
  }
}

// tslint:disable-next-line:max-classes-per-file
export class StyledLayout extends InjectStyledLayout {
  public render() {
    return (
      <InjectStyledLayout {...this.props}>
        <div ref={this.props.refDelegate}>
          {this.props.children}
        </div>
      </InjectStyledLayout>
    );
  }
}
