import * as classnames from 'classnames';
import * as React from 'react';
import { Display, InjectLayout, Padding } from '../../layout';
import './styles.scss';

export interface AnimatedGlitchLogoProps {
  width?: number;
  height?: number;
  padding?: Padding;
}

export interface AnimatedGlitchLogoState {
  bodyAnimating: boolean;
  eyesAnimating: boolean;
  animationQueue: GlitchAnimationStep[];
}

export enum GlitchAnimationStep {
  mouseEnter = 'mouse-enter',
  mouseLeave = 'mouse-leave',
  mouseDown = 'mouse-down',
  mouseUp = 'mouse-up',
}

export class AnimatedGlitchLogo extends React.Component<AnimatedGlitchLogoProps, AnimatedGlitchLogoState> {
  public state: AnimatedGlitchLogoState = {
    animationQueue: [],
    bodyAnimating: false,
    eyesAnimating: false,
  };

  public render() {
    const step = this.state.animationQueue[0];

    return (
      <InjectLayout padding={this.props.padding} display={Display.InlineFlex}>
        <div
          className={'tw-animated-glitch-logo'}
          onMouseEnter={this.onMouseEnter}
          onMouseLeave={this.onMouseLeave}
          onMouseDown={this.onMouseDown}
          onMouseUp={this.onMouseUp}
        >
          <InjectLayout display={Display.InlineFlex}>
            <figure>
              <svg
                className="tw-animated-glitch-logo__svg"
                overflow={'visible'}
                width={(this.props.width || 30) + 'px'}
                height={(this.props.height || 30) + 'px'}
                version="1.1"
                viewBox={`0 0 30 30`}
                x="0px"
                y="0px"
              >
                <g
                  onAnimationEnd={this.onBodyAnimationEnd}
                  className={classnames({
                    [`tw-animated-glitch-logo__body`]: true,
                    [`tw-animated-glitch-logo__body--${step}`]: !!step,
                  })}
                >
                  <path d="M4,7 L5.56799,3 L27,3 L27,18 L21,24 L16,24 L12.88599,27 L9,27 L9,24 L4,24 L4,7 Z M21,20 L25,16 L25,5 L8,5 L8,20 L12,20 L12,23 L15,20 L21,20 Z" />
                  <g
                    onAnimationEnd={this.onEyesAnimationEnd}
                    className={classnames({
                      [`tw-animated-glitch-logo__eyes`]: true,
                      [`tw-animated-glitch-logo__eyes--${step}`]: !!step && this.state.eyesAnimating,
                    })}
                  >
                    <polygon className="tw-animated-glitch-logo__right-eye" points="21 9 19 9 19 15 21 15" />
                    <polygon className="tw-animated-glitch-logo__left-eye" points="16 9 14 9 14 15 16 15" />
                  </g>
                </g>
              </svg>
            </figure>
          </InjectLayout>
        </div>
      </InjectLayout>
    );
  }

  private onBodyAnimationEnd = (e: React.AnimationEvent<SVGPathElement>) => {
    e.stopPropagation();

    if (this.state.animationQueue.length > 1) {
      this.setState((prevState) => ({
        // Move to next animation in queue
        animationQueue: prevState.animationQueue.slice(1),
      }));
    } else {
      this.setState({
        bodyAnimating: false,
      });
    }
  }

  private onEyesAnimationEnd = (e: React.AnimationEvent<SVGPathElement>) => {
    e.stopPropagation();

    this.setState({
      eyesAnimating: false,
    });
  }

  private onMouseEnter = () => {
    // Start this animation immediately in order to provide responsive UX
    this.queueAnimation(GlitchAnimationStep.mouseEnter, true);
  }

  private onMouseLeave = () => {
    this.queueAnimation(GlitchAnimationStep.mouseLeave);
  }

  private onMouseDown = () => {
    // Start this animation immediately in order to provide responsive UX
    this.queueAnimation(GlitchAnimationStep.mouseDown, true);
  }

  private onMouseUp = () => {
    this.queueAnimation(GlitchAnimationStep.mouseUp);
  }

  private queueAnimation(step: GlitchAnimationStep, startNow: boolean = false) {
    this.setState((prevState) => ({
      bodyAnimating: true,
      eyesAnimating: true,
      animationQueue: (prevState.bodyAnimating && !startNow) ? prevState.animationQueue.concat([step]) : [step],
    }));
  }
}
