import * as React from 'react';
import { cn } from '../../../utils/classnames';
import { getDataProps } from '../../../utils/data-props';
import { Color, CoreText } from '../../core-text';
import { Interactable, InteractableType } from '../../interactable';
import { AlignItems, Display, Layout, TextAlign, VerticalAlign } from '../../layout';
import { StyledLayout } from '../../styled-layout';
import { SVG, SVGAsset } from '../../svg';
import './styles.scss';

export enum Sorting {
  Default,
  Ascending,
  Descending,
}

export interface TableHeadingProps {
  label: string;
  onClick?: React.MouseEventHandler<HTMLElement>;
  sorting?: Sorting;
  textAlign?: TextAlign;
  verticalAlign?: VerticalAlign;
}

export const TableHeading: React.SFC<TableHeadingProps> = (props) => {
  const classes: ClassValue = {};

  let heading: JSX.Element | undefined;
  let sortArrow: JSX.Element | undefined;

  if (props.sorting !== undefined) {

    switch (Sorting[props.sorting]) {
      case 'Ascending':
        sortArrow = <SVG asset={SVGAsset.GlyphArrUp} width={10} />;
        break;

      case 'Descending':
        sortArrow = <SVG asset={SVGAsset.GlyphArrDown} width={10} />;
        break;

      default:
        sortArrow = <SVG asset={SVGAsset.GlyphArrUpDown} width={10} />;
        break;
    }

    heading = (
      <Interactable type={InteractableType.Alpha} onClick={props.onClick}>
        <StyledLayout color={Color.Link} display={Display.Flex} alignItems={AlignItems.Center} padding={1}>
          <Layout margin={{ right: 0.5 }} flexGrow={1}>
            <CoreText bold>{props.label}</CoreText>
          </Layout>
          {sortArrow}
        </StyledLayout>
      </Interactable>
    );
  } else {
    heading = (
      <Layout padding={1}>
        <CoreText bold>{props.label}</CoreText>
      </Layout>
    );
  }

  return (
    <Layout verticalAlign={props.verticalAlign} textAlign={props.textAlign} className={cn('tw-table-heading', classes)} {...getDataProps(props)}>
      {heading}
    </Layout>
  );
};

TableHeading.displayName = 'TableHeading';
