import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { CoreInteractivePublicProps } from '../core-interactive';
import { Color } from '../core-text';
import { Interactable, InteractableType } from '../interactable';
import { AlignItems, Display, Layout } from '../layout';
import { Background, BorderRadius, StyledLayout } from '../styled-layout';
import { SVG, SVGAsset } from '../svg';
import './styles.scss';

export enum TagType {
  Default = '',
  Accent = 'tw-tag--accent',
}

export enum TagAction {
  Remove = 1,
  Add,
}

export interface TagProps extends CoreInteractivePublicProps {
  action?: TagAction;
  /**
   * Display an icon within the tag. _Tags are either text-only or
   * icon-only; setting this prop will hide the `label` text.
   */
  icon?: SVGAsset;
  /**
   * Text to render within the tag.
   * @example Competitive
   */
  label: string;
  /** Set the display type of the tag. */
  type?: TagType;
}

export const Tag: React.SFC<TagProps> = (props) => {
  const classes: ClassValue = {
    'tw-tag': true,
    'tw-tag--default': !props.type,
  };

  let tabIndex: number | undefined;
  if (props.tabIndex) {
    tabIndex = props.tabIndex;
  }

  let content: JSX.Element;

  if (props.icon) {
    content = <SVG asset={props.icon} />;
  } else {
    content = <div>{props.label}</div>;
  }

  return (
    <StyledLayout
      className={cn(classes)}
      display={Display.InlineBlock}
      background={props.type ? Background.Accent : undefined}
      color={props.type ? Color.Overlay : Color.Base}
      borderRadius={BorderRadius.Medium}
    >
      <Interactable
        type={props.type ? InteractableType.Overlay : InteractableType.Alt}
        tabIndex={tabIndex}
        borderRadius={BorderRadius.Medium}
        linkTo={props.linkTo}
        onClick={props.onClick}
        disabled={props.disabled}
        targetBlank={props.targetBlank}
        download={props.download}
        ariaLabel={props.ariaLabel || (props.icon ? props.label : undefined)}
        renderLink={props.renderLink}
        title={props.title}
        {...getDataProps(props)}
      >
        <Layout
          className="tw-tag__content"
          display={Display.Flex}
          alignItems={AlignItems.Center}
        >
        {content}
        {props.action && (
          <Layout
            display={Display.Flex}
            alignItems={AlignItems.Center}
            margin={{ left: 0.5 }}
          >
            <SVG width={10} height={10} asset={props.action === TagAction.Add ? SVGAsset.Plus : SVGAsset.RemoveTag} />
          </Layout>
        )}
        </Layout>
      </Interactable>
    </StyledLayout>
  );
};

Tag.displayName = 'Tag';
