import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { newUUIDv4 } from '../../utils/unique-id';
import { Display, InjectLayout } from '../layout';
import './styles.scss';

export enum TooltipDirection {
  Top = 'tw-tooltip--up',
  Bottom = 'tw-tooltip--down',
  Left = 'tw-tooltip--left',
  Right = 'tw-tooltip--right',
}

export enum TooltipAlign {
  Left = 'tw-tooltip--align-left',
  Center = 'tw-tooltip--align-center',
  Right = 'tw-tooltip--align-right',
}

export interface TooltipProps {
  align?: TooltipAlign;
  children?: React.ReactNode;
  direction?: TooltipDirection;
  display?: Display;
  /**
   * The content of the tooltip.
   *
   * @example Viewers
   */
  label: string;
  /**
   * Manually toggle the display of the tooltip.
   *
   * @example true
   */
  id?: string;
  show?: boolean;
  width?: number;
  offsetX?: string;
  offsetY?: string;
  title?: string;
}

/**
 * Displays a text label upon hover. A tooltip should provide contextual
 * information about a given element. __Because of its reliance on the 'hover'
 * event, it is not a mobile-friendly pattern.__
 */
export const Tooltip: React.SFC<TooltipProps> = (props) => {
  const generatedId = newUUIDv4();
  const classes: ClassValue = {
    'tw-tooltip': true,
    'tw-tooltip--wrap': !!props.width,
  };

  const wrapperClasses: ClassValue = {
    'tw-tooltip-wrapper': true,
    'tw-tooltip-wrapper--show': props.show,
  };

  if (props.direction) {
    classes[props.direction] = true;
  } else {
    classes[TooltipDirection.Top] = true;
  }

  if (props.align && props.align !== TooltipAlign.Center) {
    if (props.direction !== TooltipDirection.Left && props.direction !== TooltipDirection.Right) {
      classes[props.align] = true;
    } else {
      classes[TooltipAlign.Center] = true;
    }
  } else {
    classes[TooltipAlign.Center] = true;
  }

  let marginLeft: string | undefined;
  let marginRight: string | undefined;
  let marginTop: string | undefined;
  let marginBottom: string | undefined;

  if (props.offsetX) {
    if (props.align === TooltipAlign.Right) {
      marginRight = props.offsetX;
    } else {
      marginLeft = props.offsetX;
    }
  }

  if (props.offsetY) {
    if (!props.direction || props.direction === TooltipDirection.Top) {
      marginBottom = props.offsetY;
    } else {
      marginTop = props.offsetY;
    }
  }

  const styles: React.CSSProperties = {
    marginLeft,
    marginRight,
    marginTop,
    marginBottom,
    width: props.width,
  };

  return (
    <InjectLayout display={props.display ? props.display : Display.InlineFlex} {...getDataProps(props)}>
      <div className={cn(wrapperClasses)} aria-describedby={props.id || generatedId} title={props.title}>
        {props.children}
        <div className={cn(classes)} style={styles} data-a-target="tw-tooltip-label" role="tooltip" id={props.id || generatedId}>{props.label}</div>
      </div>
    </InjectLayout>
  );
};

Tooltip.displayName = 'Tooltip';
