import * as React from 'react';
import { cn } from '../../utils/classnames';
import { getDataProps } from '../../utils/data-props';
import { FlexWrap, JustifyContent, Layout } from '../layout/index';
import './styles.scss';

export enum TowerGutter {
  Default = '',
  ExtraSmall = 'tw-tower--gutter-xs',
  Small = 'tw-tower--gutter-sm',
  Large = 'tw-tower--gutter-lg',
  None = 'tw-tower--gutter-none',
}

export enum TowerChildWidth {
  ExtraSmall = 'tw-tower--120', // 120px
  Small = 'tw-tower--180', // 180px
  Medium = 'tw-tower--240', // 240px
  Large = 'tw-tower--300', // 300px
}

export interface TowerProps {
  /** Center the children. */
  center?: boolean;
  /** The items to be laid out by tower.
   * @example
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *    <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *    <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   *  <div>
   *    <StyledLayout background={Background.Accent} padding={4} margin={{bottom: 1}} fullWidth />
   *  </div>
   */
  children?: React.ReactNode;
  /** Sets the width of the children. */
  childWidth?: TowerChildWidth;
  /** Sets the size of gutter between the children. */
  gutterSize?: TowerGutter;
  noGrow?: boolean;
  noWrap?: boolean;
  placeholderItems?: number;
}

/**
 * Tower helps to lay out items of the same width while forcing a desired width on those children via flexbox.
 */
export const Tower: React.SFC<TowerProps> = (props) => {
  const classes: ClassValue = {
    'tw-tower': true,
    'tw-tower--nogrow': props.noGrow,
  };

  if (props.gutterSize) {
    classes[props.gutterSize] = true;
  }

  if (props.childWidth) {
    classes[props.childWidth] = true;
  }

  const placeholderItems = new Array<JSX.Element>();
  if (props.placeholderItems) {
    for (let i = 0; i < props.placeholderItems; ++i) {
      placeholderItems.push((
        <div className="tw-tower__placeholder" key={i} />
      ));
    }
  }

  return (
    <Layout
      justifyContent={props.center ? JustifyContent.Center : undefined}
      flexWrap={props.noWrap ? FlexWrap.NoWrap : FlexWrap.Wrap}
      className={cn(classes)}
      {...getDataProps(props)}
    >
        {props.children}
        {placeholderItems}
    </Layout>
  );
};

Tower.displayName = 'Tower';
