import * as React from 'react';
import { AlignItems, CoreText, Display, InjectLayout, JustifyContent, Layout, StyledLayout } from '../../../index';

export interface ComponentPropConfig {
  propKey: string;
  propValues: any[]; // tslint:disable-line: no-any
}

export interface ComponentProps {
  [key: string]: any; // tslint:disable-line: no-any
}

export interface Props {
  children: React.ReactNode;
  fields: ComponentPropConfig[];
}

interface State {
}

export class CombinationGenerator extends React.Component<Props, State> {

  public render() {
    const propSets = fieldsToPropPermutations(this.props.fields);
    return (
      <div>

        <Layout>
          <CoreText>{propSets.length} combinations from options: {this.props.fields.map((item) => (`${item.propKey} [${item.propValues.length}]`)).join(', ')}</CoreText>
        </Layout>

        <Layout margin={{ y: 2 }}>
          {propSets.map(this.renderRow)}
        </Layout>

      </div>
    );
  }

  private renderRow = (currentProps: ComponentProps, index: number, allProps: ComponentProps[]) => {
    const basis = 100 / Math.max(1, React.Children.count(this.props.children));
    const items = React.Children.map(this.props.children, (child, childIndex) => {
      let newChild;
      if (React.isValidElement(child)) {
        newChild = React.cloneElement(child, currentProps);
      }

      return (
        <InjectLayout key={childIndex} padding={1} display={Display.Flex} alignItems={AlignItems.Start} justifyContent={JustifyContent.Start} flexGrow={1}>
          <div style={{ flexBasis: `${basis}%` }}>
            {newChild}
          </div>
        </InjectLayout>
      );
    });

    return (
      <React.Fragment key={index}>
        {this.getRowSetHeading(currentProps, index, allProps)}
        <StyledLayout
          display={Display.Flex}
          justifyContent={JustifyContent.Start}
          padding={1}
        >
          {items}
        </StyledLayout>
      </React.Fragment>
    );
  }

  private getRowSetHeading(currentObj: ComponentProps, index: number, allObjs: ComponentProps[]) {
    const diffProps = [];

    const next = allObjs[index + 1] || {};
    const prev = allObjs[index - 1] || {};

    for (const key in currentObj) {

      if (currentObj[key] === undefined) {
        continue;
      }

      if (currentObj[key] === false) {
        continue;
      }

      if (prev.hasOwnProperty(key) && prev[key] === currentObj[key]) {
        continue;
      }

      if (next.hasOwnProperty(key) && next[key] === currentObj[key]) {
        diffProps.push(key + ' = ' + currentObj[key]);
      }

    }
    return diffProps.join(', ');
  }
}

/**
 * Generates all possible permutations of component props, given a list of possible prop values.
 *
 * Ordering will group latter entries more closely and earlier entries further apart.
 * In other words:
 *   - put props higher in the list of you don't care to see their values next to each other.
 *   - put props lower in the list if you want their values to always be shown side by side.
 *   - props in the middle get spread out accordingly.
 *
 * Example:
 *
 * Given this input:
 * [
 *  [A1, A2, A3],
 *  [B1, B2, B3],
 * ]
 *
 * Function will output:
 *
 * A1 + B1
 * A1 + B2
 * A1 + B3
 *
 * A2 + B1
 * A2 + B2
 * A3 + B3
 *
 * A3 + B1
 * A3 + B2
 * A3 + B3
 *
 *
 * @param fields
 * @param obj Used when this function is called recursively; don't set on first call.
 */
function fieldsToPropPermutations(fields: ComponentPropConfig[], obj: ComponentProps = {}): ComponentProps[] {
  let results: ComponentProps[] = [];

  if (fields.length === 1) {
    const field = fields.shift();

    if (field) {
      field.propValues.forEach((value) => {
        results.push({ ...obj, [`${field.propKey}`]: value });
      });
    }
  }

  fields.forEach((field, index) => {
    if (index === fields.length - 1) {
      return;
    }

    if (obj.hasOwnProperty(field.propKey)) {
      return;
    }

    field.propValues.forEach((value) => {
      const nextResults = fieldsToPropPermutations(fields.slice(1), {
        ...obj,
        [`${field.propKey}`]: value,
      });

      results = results.concat(nextResults);
    });
  });

  return results;
}
