import * as React from 'react';
import { NavLink, Redirect, RedirectProps, Route, RouteProps } from 'react-router-dom';
import { Color, CoreLink, CoreText, Display, FlexDirection, FlexWrap, FontSize, FormGroup, FormGroupOrientation, InjectStyledLayout, JustifyContent, Layout, Overflow, Position, StyledLayout, SVG, SVGAsset, SVGType } from '../../../index';
import { newUUIDv4 } from '../../../utils/unique-id';
import { FeatureToggle } from '../feature-toggle';
import { NavItem, PageIndex } from './routes';
import './styles.scss';

export const MainNav = () => {
  let navLevel = 0;
  function renderNavItem(item: NavItem): JSX.Element {
    let itemContent: JSX.Element | string = item.name;
    let subItems: JSX.Element | null = null;
    const generatedId = newUUIDv4();
    navLevel++;

    if (item.linkTo) {
      itemContent = (
        <NavLink activeClassName="main-nav__link--active" className={`main-nav__link main-nav__link-nesting-depth--${navLevel}`} to={item.linkTo} exact={item.exact}>{itemContent}</NavLink>
      );
    } else {
      itemContent = (
        <Layout padding={{ top: 2, x: 3 }} margin={{ bottom: 1 }}>
          <CoreText fontSize={FontSize.Size7} color={Color.Alt2}>{itemContent}</CoreText>
        </Layout>
      );
    }
    if (item.links) {
      subItems = (
        <Layout className="main-nav__link-container">{item.links.map(renderNavItem)}</Layout>
      );
    }
    navLevel--;
    return <Layout key={generatedId}>{itemContent}{subItems}</Layout>;
  }

  return (
    <StyledLayout className="main-nav" position={Position.Fixed} display={Display.Flex} flexDirection={FlexDirection.Column} flexWrap={FlexWrap.NoWrap} fullHeight borderRight>
      <InjectStyledLayout className="main-nav__logo" display={Display.Flex} flexShrink={0} justifyContent={JustifyContent.Center} padding={{ x: 2, y: 5 }} borderBottom>
        <CoreLink to="/">
          <SVG type={SVGType.Inherit} asset={SVGAsset.LogoTwitch} width={94} height={32} />
          <h1 className="tw-hide-accessible">Core UI</h1>
        </CoreLink>
      </InjectStyledLayout>
      <Layout overflow={Overflow.Auto} flexGrow={1} flexShrink={1}>
        {PageIndex.map(renderNavItem)}
      </Layout>
      <StyledLayout padding={{ y: 2, x: 2 }} borderTop>
        <Layout margin={{ bottom: 1 }}>
          <FormGroup label="Dark Mode" orientation={FormGroupOrientation.Horizontal}>
            <FeatureToggle feature="theme-dark" />
          </FormGroup>
        </Layout>
        <Layout margin={{ bottom: 1 }}>
          <FormGroup label="Hover Feature" orientation={FormGroupOrientation.Horizontal}>
            <FeatureToggle feature="hover" />
          </FormGroup>
        </Layout>
        <FormGroup label="Touch Feature" orientation={FormGroupOrientation.Horizontal}>
          <FeatureToggle feature="touch" />
        </FormGroup>
      </StyledLayout>
    </StyledLayout>
  );
};

/**
 * Takes a NavItem or NavItem[] and creates a flat array including the links,
 * ordered by depth first traversal.
 */
function flattenPages(items: NavItem | NavItem[]): NavItem[] {
  const allPages: NavItem[] = [];
  let queue: NavItem[] = Array.isArray(items) ? items : [items];

  while (queue.length > 0) {
    // Get the next NavItem from the queue
    const navItem = queue.shift();
    if (!navItem) {
      break;
    }

    allPages.push(navItem);

    // If this NavItem has any links, move them to the front of the queue
    if (navItem.links) {
      queue = navItem.links.concat(queue);
    }
  }

  return allPages;
}

function renderRoutes(items: NavItem | NavItem[]) {
  return flattenPages(items)
    .map((item: NavItem, pageIndex: number) => {
      const isComponentGroupParent = (!!item.linkTo && !!item.links && item.links.length > 0);
      if (isComponentGroupParent) {
        const redirectProps: RedirectProps = {
          path: item.linkTo,
          to: {
            pathname: item.links![0].linkTo,
          },
        };
        const renderPage = () => <Redirect {...redirectProps} />;
        return <Route key={pageIndex} exact path={item.linkTo} render={renderPage} />;
      } else {
        const routeProps: RouteProps = {
          path: item.linkTo!,
          component: item.component!,
        };

        return <Route key={pageIndex} {...routeProps} exact />;
      }
    });
}

export const MainRoutes = () => {
  return (
    <Layout className="main-layout" padding={3} display={Display.Flex} fullHeight fullWidth flexShrink={1} flexGrow={1}>
      {PageIndex.map(renderRoutes)}
    </Layout>
  );
};
