import { NormalizedCacheObject } from 'apollo-cache-inmemory';
import { ApolloClient } from 'apollo-client';
import gql from 'graphql-tag';
import * as React from 'react';
import { OperationOption } from 'react-apollo';
import { mockSuccessResponse } from 'src/tests/utils/apollo';
import { mountWithGraphQLDataAndClient } from 'src/tests/utils/graphql';
import { withGraphQL } from './component';

interface TestValue {
  value: number;
}

interface TestData {
  a?: TestValue;
  b?: TestValue;
}

interface TestProps {
  data: TestData;
}

class TestComponent extends React.Component<TestProps> {
  public render() {
    return null;
  }
}

const query = gql`
query With_GraphQL_Test_Query {
  a {
    value
  }
  b {
    value
  }
}
`;

function createData(data: TestData) {
  const a = data.a === undefined ? undefined : {
    __typename: 'Value',
    value: 0,
    ...data.a,
  };

  const b = data.b === undefined ? undefined : {
    __typename: 'Value',
    value: 0,
    ...data.b,
  };

  return { a, b };
}

const setup = async (data: TestData, operationOptions?: OperationOption<TestProps, {}>) => {
  const mockData = mockSuccessResponse(query, {}, { data: createData(data) });
  const Test = withGraphQL(query, operationOptions)(TestComponent) as React.ComponentClass<{}>;
  const mounted = await mountWithGraphQLDataAndClient(<Test />, [mockData]);
  return mounted;
};

const writeQuery = (client: ApolloClient<NormalizedCacheObject>, data: TestData) => {
  client.writeQuery({
    query,
    data: createData(data),
  });
};

describe('withGraphQL', () => {
  describe('Corruption', () => {
    it('intercepts corrupt props when only some of the data is corrupt', async () => {
      const { wrapper, client } = await setup({
        a: { value: 42 },
        b: { value: 29 },
      });

      let node = wrapper.find(TestComponent);

      // Validate initial props from the query
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(29);

      // Write a new query with missing data
      writeQuery(client, {
        a: undefined,
        b: { value: 6 },
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate corrupt data is showing the non-corrupt value
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(6);
    });

    it('intercepts corrupt props when all of the data is corrupt', async () => {
      const { wrapper, client } = await setup({
        a: { value: 42 },
        b: { value: 29 },
      });

      let node = wrapper.find(TestComponent);

      // Validate initial props from the query
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(29);

      // Write a new query with missing data
      writeQuery(client, {
        a: undefined,
        b: undefined,
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate corrupt data is showing the non-corrupt value
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(29);
    });

    it('handles non-corrupt next props when initial props were non-corrupt', async () => {
      const { wrapper, client } = await setup({
        a: { value: 42 },
        b: { value: 29 },
      });

      let node = wrapper.find(TestComponent);

      // Validate initial props from the query
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(29);

      // Write a new query with new data
      writeQuery(client, {
        a: { value: 1 },
        b: { value: 2 },
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate we're showing the new query's props
      expect(node.props().data.a!.value).toBe(1);
      expect(node.props().data.b!.value).toBe(2);
    });

    it('handles non-corrupt next props after intercepting prior props', async () => {
      const { wrapper, client } = await setup({
        a: { value: 42 },
        b: { value: 29 },
      });

      let node = wrapper.find(TestComponent);

      // Validate initial props from the query
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(29);

      // Write a new query with missing data
      writeQuery(client, {
        a: undefined,
        b: { value: 2 },
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate corrupt data is showing the non-corrupt value
      expect(node.props().data.a!.value).toBe(42);
      expect(node.props().data.b!.value).toBe(2);

      // Write a new query with non-corrupt data
      writeQuery(client, {
        a: { value: 3 },
        b: { value: 4 },
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate we are showing the non-corrupted new query props
      expect(node.props().data.a!.value).toBe(3);
      expect(node.props().data.b!.value).toBe(4);
    });
  });

  describe('Operation Options', () => {
    it('handles when skip evaluates to true via boolean', async () => {
      const { wrapper, client } = await setup({
        a: { value: 42 },
        b: { value: 29 },
      }, {
        skip: true,
      });

      let node = wrapper.find(TestComponent);

      // Validate initial props are skipped from the query
      expect(node.props().data).toBeUndefined();

      // Write a new query with missing data
      writeQuery(client, {
        a: { value: 5 },
        b: { value: 2 },
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate props are skipped still
      expect(node.props().data).toBeUndefined();
    });

    it('handles when skip evaluates to true via function', async () => {
      const { wrapper, client } = await setup({
        a: { value: 42 },
        b: { value: 29 },
      }, {
        skip: (_props: {}) => true,
      });

      let node = wrapper.find(TestComponent);

      // Validate initial props are skipped from the query
      expect(node.props().data).toBeUndefined();

      // Write a new query with new data
      writeQuery(client, {
        a: { value: 5 },
        b: { value: 2 },
      });

      wrapper.update();
      node = wrapper.find(TestComponent);

      // Validate props are skipped still
      expect(node.props().data).toBeUndefined();
    });
  });
});
