import { DocumentNode } from 'graphql';
import * as React from 'react';
import { DataProps, graphql, GraphqlQueryControls, MutateProps, OperationOption } from 'react-apollo';
import { InferableComponentDecorator } from 'src/core/utils/inferable';
import { IndexedData } from 'src/core/utils/with-graphql/models';
import { getGuardedLoadingProps, getQueryName, getTopLevelDataProps, getTopLevelQueryFields, isDataCorrupt, shouldSkipUsingOperationOptions } from './utils';

export type IndexedGraphqlQueryControls = GraphqlQueryControls & Record<string, IndexedData>;

export type Props = Record<string, IndexedGraphqlQueryControls | {}>;

/**
 * Use this when creating a static displayName field in your HOC/wrapped component classes.
 */
export function createHOCDisplayName<P>(name: string, Component: React.ComponentType<P>) {
  return `${name}(${Component.displayName || Component.name || 'Component'})`;
}

/**
 * Allows fetching and adding data from a GraphQL server, with the added bonus
 * of protecting against corrupt data and serving the previous, non-corrupt data
 * when possible.
 *
 * We're classifying data as corrupt when Apollo will serve undefined values for
 * top-level fields in a query, when that data was previously there (and the query
 * was not set up with skips/include to cause that data to go undefined). This can
 * happen frequently in our codebase like so:
 *
 * Query A: queries a record with values 1, 2, 3 and has an id
 * Query B: queries the same record with values 5, 6, 7 and has an id
 *
 * If Query A is being displayed by a component on screen, and Query B loads with a NEW id,
 * it will populate the old data with values 5, 6, 7. Query A is listening to that change,
 * but since Query B has created a new object with values that don't fully match what A was
 * requesting, Apollo will set the data on A to undefined (since technically it's old and
 * not reconcilable with the new data). This makes it look like we're missing data in the UI.
 *
 * Instead we want to handle this scenario smartly by displaying the slightly outdated data,
 * and refetching that query so that the component can show the new data and update Apollo's
 * backing store correctly. That's what this HOC does behind the scenes.
 *
 */
export function withGraphQL<
  TProps extends TGraphQLVariables | {} = {},
  TData = {},
  TGraphQLVariables = {},
  TChildProps = Partial<DataProps<TData, TGraphQLVariables>> & Partial<MutateProps<TData, TGraphQLVariables>>
>(
  node: DocumentNode,
  operationOptions?: OperationOption<TProps, TData, TGraphQLVariables, TChildProps>,
): InferableComponentDecorator<TProps> {
  return (WrappedComponent: React.ComponentClass<TProps> | React.StatelessComponent<TProps>) => {
    class WithGraphQL extends React.Component<Props> {
      public static displayName = createHOCDisplayName(WithGraphQL.name, WrappedComponent);

      /**
       * This private instance is used to compare incoming and current props
       * for fields that the query should be populating, unless data is corrupt.
       */
      private queryFields = getTopLevelQueryFields(node);

      /**
       * Usually gql props are attached under the props.data, but devs can assign them
       * to specific names as they please.
       */
      private operationName = operationOptions && operationOptions.name || 'data';

      /**
       * This private instance is used to hold the pre-corrupt data that is merged in
       * with props when rendering (if necessary).
       */
      private cachedData: IndexedData = {};

      /**
       * Used to provide additional information when logging.
       */
      private queryName = getQueryName(node);

      /**
       * Here we want to determine if we've received corrupted data, and if so,
       * store the pre-corrupted data on the component and refetch the query.
       */
      public componentWillReceiveProps(nextProps: Readonly<Props>) {
        // If we're skipped via operationOptions, we can just skip all of our corruption/refetch logic too.
        if (shouldSkipUsingOperationOptions(nextProps, operationOptions)) {
          return;
        }

        const dataProps = getTopLevelDataProps(this.props, this.operationName, this.queryFields);
        const nextDataProps = getTopLevelDataProps(nextProps, this.operationName, this.queryFields);
        const { cachedData, operationName, queryFields, queryName } = this;

        // Note: this will also modify this.cachedData in place if corrupt data is added/removed
        const hasCorruptData = isDataCorrupt({
          dataProps,
          nextDataProps,
          cachedData,
          operationName,
          queryFields,
          queryName,
          operationOptions,
          nextProps,
        });

        // Refetch if we've encountered corrupt data
        if (hasCorruptData) {
          this.logCorruptDataFound();

          const queryProps = nextProps[this.operationName] as GraphqlQueryControls;
          if (queryProps && queryProps.refetch) {
            queryProps.refetch();
          }
        }
      }

      public render() {
        return <WrappedComponent {...this.getRenderProps()} />;
      }

      /**
       * This function will return the props with the pre-corrupted data props in this.data
       * merged in. Doing this prevents corrupted (and subsequently undefined top-level
       * query data props) data from being displayed by components when all that needs
       * to happen is the corrupted query should be refetched.
       */
      private getRenderProps() {
        const { operationName, props, queryFields, cachedData } = this;

        // If we are skipping the query, just pass props through.
        if (shouldSkipUsingOperationOptions(props, operationOptions)) {
          return props;
        }

        // We only want to alter the props if the Apollo added data prop
        // is actually an object. For instance, mutations are a function
        // so we just want to pass props through in that case.
        if (typeof props[operationName] !== 'object') {
          return props;
        }

        // While loading fresh data, Apollo can sometimes include partial query data
        // in highly sporadic conditions. Our generated schema interfaces make certain
        // guarantees about the shape of query responses, and this can result in missing
        // fields that are actually defined as non-nullable in the schema. The following
        // util is used to remove partial query results from props. If we receive
        // guarded loading props back from this util, prefer them.
        const guardedLoadingProps = getGuardedLoadingProps({
          operationName,
          operationOptions,
          props,
          queryFields,
        });

        if (guardedLoadingProps) {
          return guardedLoadingProps;
        }

        // Otherwise, return the current props and mix in cached data.
        return {
          ...props,
          [operationName]: {
            ...cachedData,
            ...props[operationName],
          },
        };
      }

      private logCorruptDataFound() {
        const data = { fieldName: this.operationName, queryName: this.queryName };
        const message = 'GraphQL data corrupted for component.';

        console.warn(message, data);
      }
    }

    // tslint:disable-next-line:no-any
    return graphql(node, operationOptions)(WithGraphQL as any) as any;
  };

}
