/**
 * Used to make TS happy when we compare values against props using indexes.
 * Normally Typescript requires well-defined interfaces, so this is necessary to handle in a general manner.
 */
export type IndexedData = Record<string, {} | undefined>;

/**
 * Used to save the fields from the query DocumentNode into a quickly usable
 * object when evaluating incoming props to determine if that is an expected
 * field or not.
 *
 * The structure is this:
 * {
 *   'fieldName': {
 *     'skip': 'variableName',
 *     'include': 'someVariableName',
 *   }
 * }
 *
 * This allows us to put any directives (skip/include) next to the field for easy evaluation
 * on incoming props to know if this value is skipped via the operationOptions or is missing
 * because it is corrupt.
 */
export type QueryFieldSet = Record<string, FieldDirectiveRecord>;

/**
 * Used as a record between skip/include -> variableName that is evaluated by the query.
 */
export type FieldDirectiveRecord = Record<string, string>;

/**
 * Used as a record between operation variable names and their computed value.
 */
export type OperationVariableRecord = Record<string, any>; // tslint:disable-line:no-any

/**
 * Used to find directives that influence whether data was included and is no
 * longer included in incoming props. this helps us understand whether data
 * is purposely missing or is corrupt. These fields correlate to @skip, @include,
 * and other directives that can be written in GQL queries.
 */
export enum DirectiveType {
  Skip = 'skip',
  Include = 'include',
}

/**
 * Used when inspecting selections while building the initial set of top-level data props
 * that this component will use to recognize when data is corrupt or not. This directly
 * relates to the kind of selection it sees when iterating: is it a field or is it a fragment
 * that we then need to inspect further.
 */
export enum SelectionKind {
  Field = 'Field',
  FragmentSpread = 'FragmentSpread',
}

/**
 * Used when inspecting the definitions of a DocumentNode to determine if we're handling
 * a fragment, or the actual query.
 */
export enum DefinitionType {
  Operation = 'OperationDefinition',
  Fragment = 'FragmentDefinition',
}
