// tslint:disable:max-file-line-count
import { ApolloError, FetchPolicy } from 'apollo-client';
import gql from 'graphql-tag';
import { GraphqlQueryControls } from 'react-apollo';
import { Props } from './component';
import { DEFAULT_UNNAMED_QUERY, fetchPolicyAllowsGuardedLoading, getGuardedLoadingProps, getOperationVariables, getQueryName, getTopLevelDataProps, getTopLevelQueryFields, GuardedLoadingOptions, isDataCorrupt, shouldSkipUsingOperationOptions } from './utils';

const mockFakeProps = (operationName: string, overrides?: GraphqlQueryControls | Record<string, {}>): Props => ({
  [operationName]: {
    error: undefined,
    networkStatus: 7,
    loading: false,
    variables: {},
    fetchMore: jest.fn(),
    refetch: jest.fn(),
    startPolling: jest.fn(),
    stopPolling: jest.fn(),
    subscribeToMore: jest.fn(),
    updateQuery: jest.fn(),
    ...overrides,
  } as GraphqlQueryControls & Record<string, {}>,
});

describe('withGraphQL Utils', () => {
  describe('Finding top-level query fields', () => {
    it('finds basic fields', () => {
      const sets = getTopLevelQueryFields(gql`
        query Test {
          a {
            value
          }
          b {
            value
          }
        }
      `);
      expect(sets).toEqual({
        a: {},
        b: {},
      });
    });

    it('finds basic fields with directives', () => {
      const sets = getTopLevelQueryFields(gql`
        query Test($val: Boolean!) {
          a @skip(if: $val) {
            value
          }
          b @include(if: $val) {
            value
          }
        }
      `);

      expect(sets).toEqual({
        a: { skip: 'val' },
        b: { include: 'val' },
      });
    });

    it('finds fragments', () => {
      const sets = getTopLevelQueryFields(gql`
        fragment testFragQuery on Query {
          a {
            value
          }
        }

        query Test {
          ...testFragQuery
          b {
            ...testFragValue
          }
        }

        fragment testFragValue on Value {
          value
        }
      `);

      expect(sets).toEqual({
        a: {},
        b: {},
      });
    });

    it('finds nested fragments', () => {
      const sets = getTopLevelQueryFields(gql`
        fragment testFragQuery on Query {
          a {
            value
          }
        }

        fragment testNestFrag on Query {
          ...testFragQuery
        }

        query Test {
          ...testNestFrag
        }
      `);

      expect(sets).toEqual({
        a: {},
      });
    });

    it('finds nested fragments intermingled with fields', () => {
      const sets = getTopLevelQueryFields(gql`
        fragment testFragQuery on Query {
          a {
            value
          }
        }

        fragment testNestFrag on Query {
          ...testFragQuery
        }

        query Test {
          ...testNestFrag
          b {
            value
          }
        }
      `);

      expect(sets).toEqual({
        a: {},
        b: {},
      });
    });

    it('applies top-level directives correctly with nested fragments', () => {
      const sets = getTopLevelQueryFields(gql`
        fragment testFragQuery on Query {
          a {
            value
          }
          b {
            value
          }
        }

        fragment testNestFrag on Query {
          ...testFragQuery
        }

        query Test($val: Boolean!) {
          ...testNestFrag @skip(if: $val)
        }
      `);

      expect(sets).toEqual({
        a: { skip: 'val' },
        b: { skip: 'val' },
      });
    });

    it('does not add mutations', () => {
      const sets = getTopLevelQueryFields(gql`
        mutation testMutation($val: Boolean!) {
          c {
            value
          }
        }

        query Test($val: Boolean!) {
          a {
            value
          }
        }
      `);

      expect(sets).toEqual({
        a: {},
      });
    });

    it('finds aliased fields', () => {
      const sets = getTopLevelQueryFields(gql`
        query Test($val: Boolean!) {
          someValue: a {
            value
          }
        }
      `);

      expect(sets).toEqual({
        someValue: {},
      });
    });

    it('finds aliased fields with directives', () => {
      const sets = getTopLevelQueryFields(gql`
        query Test($val: Boolean!) {
          someValue: a @skip(if: $val) {
            value
          }
        }
      `);

      expect(sets).toEqual({
        someValue: { skip: 'val' },
      });
    });

    it('finds aliased fields with variables', () => {
      const sets = getTopLevelQueryFields(gql`
        query Test($val: Boolean!) {
          someValue: a(val: $val) {
            value
          }
        }
      `);

      expect(sets).toEqual({
        someValue: {},
      });
    });

    it('adds multiple directives', () => {
      const sets = getTopLevelQueryFields(gql`
        query Test($val: Boolean!) {
          someValue: a @skip(if: $val) @include(if: $val) {
            value
          }
        }
      `);

      expect(sets).toEqual({
        someValue: { skip: 'val', include: 'val' },
      });
    });
  });

  describe('Getting operation variables', () => {
    it('returns a blank object when no operationOptions are available', () => {
      const variables = getOperationVariables({});
      expect(variables).toEqual({});
    });

    it('returns an object with variables if they are defined as an object inside operation options', () => {
      const variables = getOperationVariables({}, {
        options: {
          variables: {
            on: true,
            off: false,
          },
        },
      });

      expect(variables).toEqual({
        on: true,
        off: false,
      });
    });

    it('returns an object with variables if they are defined as a function inside operation options', () => {
      const variables = getOperationVariables({}, {
        options: (_: {}) => ({
          variables: {
            on: true,
            off: false,
          },
        }),
      });

      expect(variables).toEqual({
        on: true,
        off: false,
      });
    });
  });

  describe('Skipping correctly using operation options', () => {
    it('returns false when no operation options are passed in', () => {
      const shouldSkip = shouldSkipUsingOperationOptions({});
      expect(shouldSkip).toBeFalsy();
    });

    it('skips correctly when skip is a boolean propety of operation options', () => {
      const shouldSkip = shouldSkipUsingOperationOptions({}, {
        skip: true,
      });

      const shouldNotSkip = shouldSkipUsingOperationOptions({}, {
        skip: false,
      });

      expect(shouldSkip).toBeTruthy();
      expect(shouldNotSkip).toBeFalsy();
    });

    it('skips correctly when skip is a function propety of operation options', () => {
      const shouldSkip = shouldSkipUsingOperationOptions({}, {
        skip: (_: {}) => true,
      });

      const shouldNotSkip = shouldSkipUsingOperationOptions({}, {
        skip: (_: {}) => false,
      });

      expect(shouldSkip).toBeTruthy();
      expect(shouldNotSkip).toBeFalsy();
    });
  });

  describe('Finding the top-level fields that are part of Props', () => {
    it('returns an empty object if the props do not contain the operationName', () => {
      const data = getTopLevelDataProps(mockFakeProps('notData'), 'data', { value: {} });
      expect(data).toEqual({});
    });

    it('returns an empty object if the props.operationName does not contain the query field', () => {
      const data = getTopLevelDataProps(mockFakeProps('data'), 'data', { value: {} });
      expect(data).toEqual({});
    });

    it('returns any fields found in props.operationName that are in the query fields', () => {
      const data = getTopLevelDataProps(
        mockFakeProps('data', {
          value: 'Hi',
          otherValue: 'Cool',
        }),
        'data',
        { value: {} },
      );
      expect(data).toEqual({
        value: 'Hi',
      });
    });
  });

  describe('Correctly determinining corrupt data', () => {

    it('does not mark data as corrupt if no fields are missing between dataProps and nextDataProps', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = { value: 'Hi' };
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: {} },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does mark data as corrupt if fields are missing between dataProps and nextDataProps, with no skip/include logic', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: {} },
      });

      expect(hasCorruptData).toBeTruthy();
      expect(cachedData).toEqual({ value: 'Hi' });
    });

    it('does not mark data as corrupt if fields are missing because they were skipped', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { skip: 'shouldSkip' } },
        operationOptions: {
          options: {
            variables: { shouldSkip: true },
          },
        },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does mark data as corrupt if fields are missing when skip evaluates to false', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { skip: 'shouldSkip' } },
        operationOptions: {
          options: {
            variables: { shouldSkip: false },
          },
        },
      });

      expect(hasCorruptData).toBeTruthy();
      expect(cachedData).toEqual({ value: 'Hi' });
    });

    it('does not mark data as corrupt if fields are missing because they are no longer included', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { include: 'shouldInclude' } },
        operationOptions: {
          options: {
            variables: { shouldInclude: false },
          },
        },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does mark data as corrupt if fields are missing when skip evaluates to false', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { include: 'shouldInclude' } },
        operationOptions: {
          options: {
            variables: { shouldInclude: true },
          },
        },
      });

      expect(hasCorruptData).toBeTruthy();
      expect(cachedData).toEqual({ value: 'Hi' });
    });

    it('does not mark data as corrupt if fields are skipped due to { include: false, skip: true } directives - skips', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { include: 'shouldInclude', skip: 'shouldSkip' } },
        operationOptions: {
          options: {
            variables: { shouldInclude: false, shouldSkip: true },
          },
        },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does not mark data as corrupt if fields are skipped due to { include: false, skip: false } directives - does not include', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { include: 'shouldInclude', skip: 'shouldSkip' } },
        operationOptions: {
          options: {
            variables: { shouldInclude: false, shouldSkip: false },
          },
        },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does not mark data as corrupt if fields are skipped due to { include: true, skip: true } directives - skips', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { include: 'shouldInclude', skip: 'shouldSkip' } },
        operationOptions: {
          options: {
            variables: { shouldInclude: true, shouldSkip: true },
          },
        },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does mark data as corrupt if fields are not skipped due to { include: true, skip: false } directives', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data'),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: { include: 'shouldInclude', skip: 'shouldSkip' } },
        operationOptions: {
          options: {
            variables: { shouldInclude: true, shouldSkip: false },
          },
        },
      });

      expect(hasCorruptData).toBeTruthy();
      expect(cachedData).toEqual({ value: 'Hi' });
    });

    it('does not mark data as corrupt if there is an ApolloError', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data', { error: new ApolloError({}) }),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: {} },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });

    it('does not mark data as corrupt if it is loading', () => {
      const dataProps = { value: 'Hi' };
      const nextDataProps = {};
      const cachedData = {};

      const hasCorruptData = isDataCorrupt({
        dataProps,
        nextDataProps,
        nextProps: mockFakeProps('data', { loading: true }),
        operationName: 'data',
        cachedData,
        queryName: 'Test',
        queryFields: { value: {} },
      });

      expect(hasCorruptData).toBeFalsy();
      expect(cachedData).toEqual({});
    });
  });

  describe('Getting query name', () => {
    it('finds the name of a query', () => {
      const name = getQueryName(gql`
        query Test {
          id
        }
      `);
      expect(name).toBe('Test');
    });

    it('returns default for operations that are not queries', () => {
      const mutation = getQueryName(gql`
        mutation Test {
          id
        }
      `);

      const fragment = getQueryName(gql`
        fragment test on Test {
          id
        }
      `);

      expect(mutation).toBe(DEFAULT_UNNAMED_QUERY);
      expect(fragment).toBe(DEFAULT_UNNAMED_QUERY);
    });
  });

  describe('fetchPolicyAllowsGuardedLoading()', () => {
    const fetchPolicies: Record<string, boolean> = {
      'cache-first': true,
      'cache-and-network': false,
      'network-only': true,
      'cache-only': true,
      'no-cache': true,
      'standby': true,
    };

    it('returns true when no operation options are present', () => {
      expect(fetchPolicyAllowsGuardedLoading({})).toBeTruthy();
    });

    it('returns true when no fetchPolicy is defined in operation options', () => {
      expect(fetchPolicyAllowsGuardedLoading({}, {
        options: {},
      })).toBeTruthy();
    });

    for (const fetchPolicy of Object.keys(fetchPolicies)) {
      const value = fetchPolicies[fetchPolicy];

      it(`returns ${value} when using static operation options for a fetchPolicy of ${fetchPolicy}`, () => {
        const result = fetchPolicyAllowsGuardedLoading({}, {
          options: { fetchPolicy: fetchPolicy as FetchPolicy },
        });
        expect(result).toBe(value);
      });

      it(`returns ${value} when using dynamic operation options for a fetchPolicy of ${fetchPolicy}`, () => {
        const result = fetchPolicyAllowsGuardedLoading({}, {
          options: (_) => ({ fetchPolicy: fetchPolicy as FetchPolicy }),
        });
        expect(result).toBe(value);
      });
    }
  });

  describe('getGuardedLoadingProps()', () => {
    interface Data {
      user: { id: string };
      requestInfo: { countryCode: string };
      networkStatus: number;
    }

    const getProps = (networkStatus: number) => ({
      data: {
        user: { id: '5' },
        requestInfo: { countryCode: 'US' },
        networkStatus,
      },
      anotherProp: 'test',
    });

    function setup<A, B, C, D>(overrides?: Partial<GuardedLoadingOptions<A, B, C, D>>) {
      return getGuardedLoadingProps({
        operationName: 'data',
        queryFields: {
          user: {},
          requestInfo: {},
        },
        props: getProps(1),
        ...overrides,
      });
    }

    it('only removes top-level query fields when networkStatus is 1', () => {
      const nextProps = setup();
      const data = nextProps!.data as Data;
      expect(data.user).toBeUndefined();
      expect(data.requestInfo).toBeUndefined();
      expect(data.networkStatus).toBe(1);
      expect(nextProps!.anotherProp).toBeDefined();
    });

    it('returns null for a non-1 networkStatus', () => {
      const nextProps = setup({ props: getProps(2) });
      expect(nextProps).toBeNull();
    });

    it('returns null when fetchPolicy does not allow guarded loading', () => {
      const nextProps = setup({
        operationOptions: {
          options: { fetchPolicy: 'cache-and-network' },
        },
      });
      expect(nextProps).toBeNull();
    });

    it('returns null when no Apollo data props exist', () => {
      const nextProps = setup({ operationName: 'test' });
      expect(nextProps).toBeNull();
    });
  });
});
