import { DocumentNode, FieldNode, FragmentDefinitionNode, NameNode, OperationDefinitionNode, SelectionNode, VariableNode } from 'graphql';
import * as React from 'react';
import { OperationOption, QueryOpts } from 'react-apollo';
import { IndexedGraphqlQueryControls, Props } from './component';
import { DefinitionType, DirectiveType, FieldDirectiveRecord, IndexedData, OperationVariableRecord, QueryFieldSet, SelectionKind } from './models';

export const DEFAULT_UNNAMED_QUERY = 'UnnamedQuery';

/**
 * This function will analyze the query in its AST format to determine what top-level
 * fields are included. We do this to have a known set of fields to inspect incoming
 * and current props with when verifying that old props had certain keys while incoming
 * props do or do not.
 */
export function getTopLevelQueryFields(query: DocumentNode): QueryFieldSet {
  return query.definitions.reduce((sets: QueryFieldSet, def) => {
    // Pass value through since the definition is not an OperationDefinitionNode, the kind queries map to.
    if (def.kind !== DefinitionType.Operation) {
      return sets;
    }

    // Pass value through since the definition is not a query, since we don't want to influence mutations
    const operation = def as OperationDefinitionNode;
    if (operation.operation !== 'query') {
      return sets;
    }

    // Iterate through all top-level selections and add to the QueryFieldSet if necessary
    for (const selection of operation.selectionSet.selections) {
      const fieldsToAdd = getTopLevelFieldsFromSelection(selection, query);

      // Apply the field to the QueryFieldSets being reduced if we have fieldsToAdd
      if (fieldsToAdd) {
        sets = {
          ...sets,
          ...fieldsToAdd,
        };
      }
    }

    return sets;
  }, {});
}

/**
 * This function will look at the SelectionNode and find any conditional variables that apply
 * to the top-level fields it describes. Since you can conditionally skip/include a fragment
 * on Query, we have to traverse through the fragment definition in some cases to correctly
 * apply that directive to each top-level field the inner-most fragment defines.
 */
function getTopLevelFieldsFromSelection(selection: SelectionNode, query: DocumentNode) {
  // Get the field names from this selection. If this is just a top-level field, it will
  // be an array of one name, but if it is a fragment, it can map to multiple fields
  const fieldNames = getSelectionFieldNames(selection, query);

  // If, for some reason, we don't have any field names after inspecting, just return a blank object
  if (!fieldNames || fieldNames.length === 0) {
    return {} as QueryFieldSet;
  }

  // The directives like skip/include that can add information to whether
  // a value is missing on purpose or because it is corrupt.
  const directives: FieldDirectiveRecord = {};

  // Find the directives that match skip/include, based on a single variable's boolean existence.
  if (selection.directives) {
    for (const directive of selection.directives) {
      // If the directive is not a skip/include, ignore it.
      const name = directive.name.value;
      if (name !== DirectiveType.Include && name !== DirectiveType.Skip) {
        continue;
      }

      // If the directive is malformed and has no argument, ignore it.
      if (!directive.arguments) {
        continue;
      }

      // If the directive has an argument, but it's not an if, ignore it.
      const arg = directive.arguments[0];
      if (arg.name.value !== 'if') {
        continue;
      }

      // Get the variable name, and store a record of skip/include -> variableName
      const variable = (arg.value as VariableNode).name.value;
      directives[name] = variable;
    }
  }

  // Store the directives next to each field returned from this selection
  const fieldSet: QueryFieldSet = {};
  for (const field of fieldNames) {
    fieldSet[field] = directives;
  }

  return fieldSet;
}

/**
 * This function will look at a selection and determine what fieldName or names that maps to. If the selection
 * is a single field, like currentUser, this will be an array of ['currentUser']. If for some reason, the selection
 * is a fragment on Query, which you should never do, we need to recur through that fragments selections to find
 * the actual top-level field names. The fragments are defined in the DocumentNode.definitions array.
 */
function getSelectionFieldNames(selection: SelectionNode, query: DocumentNode): string[] | undefined {
  // This is a FieldNode, so we only care about its name or its alias.
  if (selection.kind === SelectionKind.Field) {
    return [selection.alias ? (selection.alias as NameNode).value : (selection as FieldNode).name.value];
  }

  // If this is a FragmentSpread, we want to inspect the OperationDefinitions to find
  // the fragment that matches, and then go through its selections to find all of the
  // top-level field names. We recur here, since you can nest fragments all the way down :(
  if (selection.kind === SelectionKind.FragmentSpread) {
    const fragmentName = selection.name.value;
    const fragmentDefinition = query.definitions.find((def) => def.kind === DefinitionType.Fragment && def.name.value === fragmentName) as FragmentDefinitionNode;

    if (!fragmentDefinition) {
      return;
    }

    // Recurse through the selections, finding all top-level fields from this fragment or every fragment it wraps.
    return fragmentDefinition.selectionSet.selections.reduce((fields: string[], sel) => {
      const fieldNames = getSelectionFieldNames(sel, query);
      if (fieldNames) {
        return fields.concat(fieldNames);
      }
    }, []);
  }
}

/**
 * This function will return a record of the variables assigned in operationOptions and what their
 * values resolve to with the props argument. This is used to reconcile whether a skip/include has
 * removed fields from the nextProps's instead of being corrupt.
 */
export function getOperationVariables<TProps, TData, TGraphQLVariables, TChildProps>(props: Readonly<Props>, operationOptions?: OperationOption<TProps, TData, TGraphQLVariables, TChildProps>): OperationVariableRecord {
  if (!operationOptions || !operationOptions.options) {
    return {};
  }

  if (typeof operationOptions.options === 'object') {
    return operationOptions.options.variables as OperationVariableRecord || {};
  } else if (typeof operationOptions.options === 'function') {
    return operationOptions.options(props as {} as TProps).variables as OperationVariableRecord || {};
  }

  return {};
}

/**
 * This function will determine if the entire query is skipped via the operation options
 * and the props passed in as an argument. When this happens we don't want to do anything
 * but pass props through since GQL is being skipped all-together at this point.
 */
export function shouldSkipUsingOperationOptions<TProps, TData, TGraphQLVariables, TChildProps>(props: Readonly<Props>, operationOptions?: OperationOption<TProps, TData, TGraphQLVariables, TChildProps>) {
  if (operationOptions && operationOptions.skip) {
    if (typeof operationOptions.skip === 'boolean') {
      return operationOptions.skip;
    } else if (typeof operationOptions.skip === 'function') {
      return operationOptions.skip(props);
    }
  }

  return false;
}

/**
 * This function will find all of the top-level data props from the query that
 * are also included in props.data. If a field is corrupted it won't be present
 * on this object.
 */
export function getTopLevelDataProps(props: Readonly<Props>, operationName: string, queryFields: QueryFieldSet): IndexedData {
  if (!props[operationName]) {
    return {};
  }

  const data: IndexedData = {};

  for (const key of Object.keys(props[operationName])) {
    if (queryFields[key]) {
      data[key] = (props[operationName] as IndexedData)[key];
    }
  }

  return data;
}

interface DataCorruptionOptions<TProps, TData, TGraphQLVariables, TChildProps> {
  dataProps: IndexedData;
  nextDataProps: IndexedData;
  nextProps: Props;
  cachedData: IndexedData;
  operationName: string;
  queryFields: QueryFieldSet;
  queryName: string;
  operationOptions?: OperationOption<TProps, TData, TGraphQLVariables, TChildProps>;
}

/**
 * This function looks at current top-level query field data compared to incoming top-level query field data
 * to determine if that data is corrupt and needs to be cached. This function modifies the passed in
 * cachedData argument and will return true/false based on if data is corrupt or not.
 *
 * Conditions that indicate corrupt data:
 * - top-level properties on props.data that were previously defined but no longer are &&
 * - we aren't currently loading &&
 * - we aren't in an errored state
 */
export function isDataCorrupt<TProps, TData, TGraphQLVariables, TChildProps>(options: DataCorruptionOptions<TProps, TData, TGraphQLVariables, TChildProps>) {
  const { dataProps, nextDataProps, nextProps, cachedData, operationName, operationOptions, queryFields, queryName } = options;
  let hasCorruptData = false;
  let hasResolvedData = false;

  // Validate if fields are corrupt, or if previously corrupt fields have been remedied.
  for (const key of Object.keys(dataProps)) {
    if (
      dataProps[key] !== undefined
      && nextDataProps[key] === undefined
      && nextProps[operationName]
      && !(nextProps[operationName] as IndexedGraphqlQueryControls).loading
      && !(nextProps[operationName] as IndexedGraphqlQueryControls).error
    ) {
      // At this point we're missing a key that used to exist. We need to determine
      // if it was skipped on purpose or if the data is corrupt.
      const skipVar = queryFields[key][DirectiveType.Skip];
      const includeVar = queryFields[key][DirectiveType.Include];
      const operationVariables = getOperationVariables(nextProps, operationOptions);

      if (skipVar && operationVariables[skipVar] !== undefined && operationVariables[skipVar]) {
        // We are skipping this top-level field, the data is not corrupt
        continue;
      } else if (includeVar && operationVariables[includeVar] !== undefined && !operationVariables[includeVar]) {
        // We are no longer including this top-level field, the data is not corrupt
        continue;
      }

      // If we've reached this point, the data is corrupt. Store the old value,
      // and flip the flag to true so that we know to try refetching the query.
      hasCorruptData = true;
      cachedData[key] = dataProps[key];
    } else if (nextDataProps[key] !== undefined && cachedData[key] !== undefined) {
      // At this point, our nextProps has values for the fields that we previously determined
      // were corrupt, so let's clear our cache for this key.
      delete cachedData[key];
      hasResolvedData = true;
    }
  }

  return hasCorruptData;
}

/**
 * This function is used to provide extra information when logging information inside
 * withGraphQL to make it easier to find queries that are consistently going corrupt.
 */
export function getQueryName(node: DocumentNode) {
  const query = node.definitions.find((def) => def.kind === DefinitionType.Operation && def.operation === 'query') as OperationDefinitionNode;
  return query && query.name && query.name.value || DEFAULT_UNNAMED_QUERY;
}

/**
 * Used to aid in determining if a fetchPolicy is explicitly set that allows
 * us to guard against Apollo providing partial query props during a fresh load.
 */
export function fetchPolicyAllowsGuardedLoading<TProps, TData = {}, TGraphQLVariables = {}, TChildProps = {}>(props: TProps, operationOptions?: OperationOption<TProps, TData, TGraphQLVariables, TChildProps>) {
  if (!operationOptions || !operationOptions.options) {
    return true;
  }

  let fetchPolicy = undefined;
  if (typeof operationOptions.options === 'function') {
    fetchPolicy = (operationOptions.options(props) as QueryOpts<TGraphQLVariables>).fetchPolicy;
  } else {
    fetchPolicy = (operationOptions.options as QueryOpts<TGraphQLVariables>).fetchPolicy;
  }

  return fetchPolicy !== 'cache-and-network';
}

export interface GuardedLoadingOptions<TProps, TData, TGraphQLVariables, TChildProps> {
  operationName: string;
  operationOptions?: OperationOption<TProps, TData, TGraphQLVariables, TChildProps>;
  props: Props & { children?: React.ReactNode };
  queryFields: QueryFieldSet;
}

/**
 * This function should be used to defend against Apollo providing partial
 * query results in props. If the networkStatus is 1 (loading fresh) and
 * the fetchPolicy is not explicitly set to 'cache-and-network', we remove
 * all data fields from the downstream props, then return the new props.
 * This will return null otherwise.
 */
export function getGuardedLoadingProps<TProps extends { children?: React.ReactNode}, TData = {}, TGraphQLVariables = {}, TChildProps = {}>(options: GuardedLoadingOptions<TProps, TData, TGraphQLVariables, TChildProps>) {
  const { operationName, operationOptions, props, queryFields } = options;
  const gqlQueryControls = props[operationName] as IndexedGraphqlQueryControls;

  if (
    gqlQueryControls
    && gqlQueryControls.networkStatus === 1
    && fetchPolicyAllowsGuardedLoading(props as TProps, operationOptions)
  ) {
    const guardedQueryControls = { ...gqlQueryControls };
    const dataKeys = Object.keys(queryFields);
    for (const field of dataKeys) {
      delete guardedQueryControls[field];
    }

    return {
      ...props,
      [operationName]: guardedQueryControls,
    };
  }

  return null;
}
