import { InMemoryCache, NormalizedCacheObject } from 'apollo-cache-inmemory';
import { ApolloClient } from 'apollo-client';
import { FetchResult, GraphQLRequest } from 'apollo-link';
import { mount, MountRendererProps, ReactWrapper } from 'enzyme';
import { ReactElement } from 'react';
import * as React from 'react';
import { ApolloProvider, GraphqlQueryControls } from 'react-apollo';
import { MockedProviderProps, MockedProviderState, MockedResponse as ReactApolloMockedResponse, mockSingleLink } from 'react-apollo/test-utils';
import { Provider, Store } from 'react-redux';
import * as renderer from 'react-test-renderer';
import { TestRendererOptions } from 'react-test-renderer';
import { fragmentMatcher } from 'src/core/utils/apollo';
import { pause } from 'src/tests/utils/pause';

// The MockedResponse in react-apollo/test-utils has been updated to require the newData property.
// None of the mocks we have written has this field so rather than update all the existing mocks
// for no reason, I decided to proxy the actual interface with this backwards compatible version.
export interface MockedResponse {
  request: GraphQLRequest;
  result?: FetchResult;
  error?: Error;
  delay?: number;
  newData?: () => GraphQLRequest;
}

/**
 * Helper for testing elements that rely on GraphQL data.
 * Uses react-apollo <MockedProvider> HOC to provide data props that
 * mock those provided by react-apollo Provider HOC.
 * Applies the passed-in testRenderer function to the <MockedProvider>-wrapped JSX Element and returns a promise,
 * which resolves with a rendered version of the passed-in element.  Depending on the mock provided, this component
 * will reflect all data props loaded, or an error, as if the first request had just completed (networkStatus 7 or 8).
 */

type AnyRenderOptions = MountRendererProps | TestRendererOptions;

// tslint:disable-next-line:no-any
function safeToUpdate(wrapper: any, testRenderer: Function) {
  return wrapper.update && testRenderer === mount;
}

/**
 * Usable on its own if you want to mount a GQL decorated component and check its loading state
 */
export function wrapWithGQLProvider(element: JSX.Element, mocks: MockedResponse[]) {
  const client = new ApolloClient({
    // See the comment at the top of the file regarding why this blind cast is necessary
    link: mockSingleLink(...mocks as ReactApolloMockedResponse[]),
    cache: new InMemoryCache({ fragmentMatcher }),
  });

  // tslint:disable:no-any
  return {
    wrappedComponent: (
      <TwilightMockedProvider mocks={mocks} client={client}>
        <Provider store={{} as Store<any>}>
          {element}
        </Provider>
      </TwilightMockedProvider>
    ),
    client,
  };
  // tslint:enable:no-any
}

function getGQLRenderable(element: JSX.Element, mocks: MockedResponse[], testRenderer: Function, renderOptions?: AnyRenderOptions) {
  jest.useFakeTimers();

  const { wrappedComponent, client } = wrapWithGQLProvider(element, mocks);

  const wrapper = testRenderer(
    wrappedComponent,
    renderOptions,
  );

  jest.runOnlyPendingTimers();
  jest.useRealTimers();

  return {
    wrapper,
    client,
  };
}

function withGraphQLData<T>(element: JSX.Element, mocks: MockedResponse[], testRenderer: Function, renderOptions?: AnyRenderOptions) {
  const { wrapper } = getGQLRenderable(element, mocks, testRenderer, renderOptions);

  return new Promise<T>(async (resolve) => {
    await pause();
    setImmediate(() => {
      if (safeToUpdate(wrapper, testRenderer)) {
        wrapper.update();
      }
      resolve(wrapper);
    });
  });
}

interface WithGQLElementAndClient<T> {
  wrapper: T;
  client: ApolloClient<NormalizedCacheObject>;
}

function withGraphQLDataAndClient<T>(element: JSX.Element, mocks: MockedResponse[], testRenderer: Function, renderOptions?: AnyRenderOptions) {
  const { wrapper, client } = getGQLRenderable(element, mocks, testRenderer, renderOptions);

  return new Promise<WithGQLElementAndClient<T>>((resolve) => {
    setImmediate(() => {
      if (safeToUpdate(wrapper, testRenderer)) {
        wrapper.update();
      }
      resolve({
        wrapper,
        client,
      });
    });
  });
}

/**
 * Uses our GraphQL mocking helper to return an enzyme mounted wrapper with mocked
 * GraphQL responses applied.
 */
export function mountWithGraphQLData<P>(element: ReactElement<P>, mocks: MockedResponse[], renderOptions?: MountRendererProps) {
  return withGraphQLData<ReactWrapper<P, {}>>(element, mocks, mount, renderOptions);
}

/**
 * Uses our GraphQL mocking helper to return an enzyme mounted wrapper with mocked
 * GraphQL responses applied as well as the apollo client to influence the store in
 * tests as you please.
 */
export function mountWithGraphQLDataAndClient<P>(element: ReactElement<P>, mocks: MockedResponse[], renderOptions?: MountRendererProps) {
  return withGraphQLDataAndClient<ReactWrapper<P, {}>>(element, mocks, mount, renderOptions);
}

/**
 * Uses our GraphQL mocking helper to return markup from react-test-renderer with mocked
 * GraphQL responses applied.
 */
export function renderWithGraphQLData<P>(element: ReactElement<P>, mocks: MockedResponse[], renderOptions?: TestRendererOptions) {
  return withGraphQLData<renderer.ReactTestRenderer>(element, mocks, renderer.create, renderOptions);
}

/**
 * Useful for mocking GraphQL's "data" prop in shallow render testing.
 */
export function mockGraphQLDataProps<Data>(
  queryResponse: Data,
  dataProps?: Partial<GraphqlQueryControls>,
): GraphqlQueryControls & Data {
  return Object.assign({
    networkStatus: 7,
    loading: false,
    error: undefined,
    variables: {},
    fetchMore: jest.fn(),
    refetch: jest.fn(),
    startPolling: jest.fn(),
    stopPolling: jest.fn(),
    subscribeToMore: jest.fn(),
    updateQuery: jest.fn(),
  }, queryResponse, dataProps);
}

type TwilightMockedProviderProps = MockedProviderProps & { client: ApolloClient<NormalizedCacheObject> };

/**
 * Lifted from react-apollo's implementation but modified to accept a client prop
 * which they conveniently removed from the latest react-apollo version bump.
 */
export class TwilightMockedProvider extends React.Component<TwilightMockedProviderProps, MockedProviderState> {
  public static defaultProps: MockedProviderProps = {
    addTypename: true,
  };

  constructor(props: TwilightMockedProviderProps) {
    super(props);
    const { client } = this.props;
    this.state = { client };
  }

  public componentDidMount() {
    this.flushApolloQueries();
  }

  public componentWillUnmount() {
    this.flushApolloQueries();
  }

  public render() {
    return <ApolloProvider client={this.state.client}>{this.props.children}</ApolloProvider>;
  }

  private flushApolloQueries() {
    if (!this.state.client.queryManager) {
      return;
    }
    const scheduler = this.state.client.queryManager.scheduler;
    Object.keys(scheduler.registeredQueries).forEach((queryId) => {
      scheduler.stopPollingQuery(queryId);
    });
    Object.keys(scheduler.intervalQueries).forEach((interval: string) => {
      scheduler.fetchQueriesOnInterval(+interval);
    });
  }
}
