const db = require("./helpers/dynamoDBHelpers.js");
const gameLogic = require("sumo-game-logic");
const commonHelpers = require("./helpers/commonHelpers.js");
const logger = require("./logger.js");

/** Initializes a game by creating the default game state. Returns true for success,
 * false for failure. */
module.exports.initializeGame = async function(channelID, gameID, playersMetadata) {
    logger.info(`Initializing game on channel ${channelID} with gameID ${gameID}.`);

    const initialDynamoGameState = gameLogic.DynamoTypes.createInitialDynamoGameState(gameID, playersMetadata);

    const dbResponse = await db.dynamoPut(db.DynamoTables.LiveGames, {
        gameID: gameID,
        channelID: channelID,
        storedState: initialDynamoGameState
    });

    if (!dbResponse.success) {
        logger.error("An error occurred when creating a new entry in LiveGames.");
        return false;
    }

    setTimeout(() => {calculateNewState(gameID, channelID)}, gameLogic.Constants.FIRST_ROUND_INPUT_PHASE_DURATION_SEC*1000);
    return true;
}

/** Reads the old state and inputs from DynamoDB, calculates the new state, and stores it in DynamoDB.
 * Returns true for success, false for failure. */
async function calculateNewState(gameID, channelID) {
    // Get the state for last round.
    const stateResponse = await commonHelpers.getGameState(gameID);

    if (stateResponse.error || stateResponse.gameState == null) {
      logger.error(`An error occurred while calculating new game state for gameID ${gameID} on channel ${channelID}. The current state was not obtained successfully.`);
      return false;
    }

    const oldState = stateResponse.gameState.storedState;

    // Get the inputs for this round.
    const currentRound = stateResponse.gameState.storedState.roundNumber + 1;
    const inputResponse = await commonHelpers.getPlayerInputs(gameID, currentRound);
    
    if (inputResponse.error) {
      logger.error(`An error occurred while calculating new game state for gameID ${gameID} on channel ${channelID}. Unable to get the player inputs for the current round.`);
      return false;
    }

    // Retrieve and update the latest game state.
    const newState = gameLogic.calculateNewState(oldState, inputResponse.inputs);
    const updateGameStateSuccess = await updateGameState(gameID, newState);

    if (!updateGameStateSuccess) {
        logger.error(`An error occurred while calculating new game state for gameID ${gameID} on channel ${channelID}. Updating the game state was unsuccessful.`);
        return false;
    }

    // Only set the timeout to calculate the next round if there is no winner. 
    if (newState.gameWinnerMetadata != null) {
        logger.info(`Game ${gameID} on channel ${channelID} has ended with winner opaqueUserID: ${newState.gameWinnerMetadata.opaqueUserID}. The results will be recorded after a short delay.`);
    }
    else {
        setTimeout(() => {calculateNewState(gameID, channelID)}, gameLogic.Constants.FULL_ROUND_DURATION_SEC*1000);
    }

    return true;
}

/** Updates the game state for the provided gameID to the provided state. Returns
 * true for success, false for failure. */
async function updateGameState(gameID, newDynamoGameState) {
    const dbResponse = await db.dynamoUpdate(
        db.DynamoTables.LiveGames,
        {gameID: gameID},
        {storedState: newDynamoGameState}
    );
    return dbResponse.success;
}