var db = require("./dynamoDBHelpers.js");
const constants = require("./constants.js");
const logger = require("../logger.js");

module.exports.getISOTimestamp = function () {
  return new Date().toISOString();
}

module.exports.addSessionHistoryEntry = async function(sessionHistoryEntry) {
  const newSessionHistoryResponse = await db.dynamoPut(db.DynamoTables.SessionHistory, sessionHistoryEntry);

  if (!newSessionHistoryResponse.success) {
    logger.error(`An error occurred while adding a new entry to the SessionHistory table. The attempted new entry was ${JSON.stringify(sessionHistoryEntry)}`);
    return false;
  }

  return true;
}

module.exports.getChannelWhitelistStatus = async function(channelID) {
  const response = {
    isWhitelisted: false,
    error: false
  };

  const dbResponse = await db.dynamoGet(db.DynamoTables.WhitelistedChannels, {
    channelID: channelID
  });

  if (!dbResponse.success) {
    logger.error(`An error occurred while getting the whitelist status for channel ${channelID}.`);
    response.error = true;
    return response;
  }

  response.isWhitelisted = dbResponse.itemFound;

  return response;
}

module.exports.channelHasOngoingSession = async function(channelID) {
    const response = {
      result: false,
      error: false
    };

    if (!channelID) {
        return response;
    }

    const dbResponse = await db.dynamoGet(db.DynamoTables.LiveSessions, {
      channelID: channelID
    });
  
    if (!dbResponse.success) {
      logger.error(`An error occurred when reading from LiveSessions for channel ${channelID}.`);
      response.error = true;
      return response;
    }
  
    response.result = dbResponse.itemFound

    return response;
}
  
module.exports.getUserSessionState = async function(channelID, opaqueUserID) {
    const response = {
      matchmaking: false,
      matchmakingStartTimestamp: constants.NULL_TIMESTAMP,
      gameID: constants.NO_GAME_ID,
      error: false
    };

    if (!channelID || !opaqueUserID) {
      logger.error(`Improper arguments for getUserSessionState: channelID: ${channelID}, opaqueUserID: ${opaqueUserID}. Both channelID and opaqueUserID are required.`);
      return response;
    }
  
    const dbResponse = await db.dynamoGet(db.DynamoTables.LivePlayers, {
      channelID: channelID,
      opaqueUserID: opaqueUserID,
    });
  
    if (!dbResponse.success) {
      logger.error(`An error occurred when reading from LivePlayers for channelID ${channelID}, opaqueUserID: ${opaqueUserID}.`);
      response.error = true;
      return response;
    }
  
    if (!dbResponse.itemFound) {
      logger.info(`Unable to find channelID: ${channelID}, opaqueUserID: ${opaqueUserID}.`);
      return response;
    }
  
    response.matchmakingStartTimestamp = dbResponse.item.matchmakingStartTimestamp;
    response.matchmaking = response.matchmakingStartTimestamp != null && response.matchmakingStartTimestamp != constants.NULL_TIMESTAMP;
    response.gameID = dbResponse.item.gameID;
  
    return response;
}
  
module.exports.getGameState = async function(gameID) {
    const response = {
      gameState: null,
      error: false  
    }
    if (!gameID) {
      return response;
    }
  
    const dbResponse = await db.dynamoGet(db.DynamoTables.LiveGames, {
      gameID: gameID
    });

    if (!dbResponse.success) {
      logger.error(`An error occurred when getting gameID ${gameID} from LiveGames.`);
      response.error = true;
      return response;
    }
  
    if (!dbResponse.itemFound) {
      logger.info(`No game state was found for gameID ${gameID}.`);
      return response;
    }
  
    response.gameState = dbResponse.item;
    return response;
}

/** Gets all the inputs for the provided gameID for the specified round. */
module.exports.getPlayerInputs = async function(gameID, roundNumber) {
  const response = {
    inputs: null,
    error: false  
  }
  if (!gameID || !roundNumber) {
    logger.error(`Improper arguments for getPlayerInputs: gameID: ${gameID}, roundNumber: ${roundNumber}.`);
    response.error = true;
    return response;
  }

  const filterExpression = 'gameID = :id AND roundNumber = :roundNum';
  const expressionAttributeValues = {':id' : gameID, ':roundNum' : roundNumber};
  const dbResponse = await db.dynamoScan(db.DynamoTables.LivePlayerInputs, filterExpression, expressionAttributeValues);

  if (!dbResponse.success) {
    logger.error(`Unable retrieve the list of player inputs from the LivePlayerInputs table for gameID ${gameID}, roundNumber: ${roundNumber}.`);
    response.error = true;
    return response;
  }

  response.inputs = dbResponse.items;
  return response;
}

module.exports.getNumActivePlayersInChannel = async function(channelID) {
  const response = {
    numPlayers: 0,
    error: false
  };

  if (!channelID) {
      return response;
  }

  const filterExpression = 'channelID = :id';
  const expressionAttributeValues = {':id' : channelID};
  const dbResponse = await db.dynamoScan(db.DynamoTables.LivePlayers, filterExpression, expressionAttributeValues);

  if (!dbResponse.success) {
    logger.error(`Unable to retrieve the number of active players from the LivePlayers table for channel ${channelID}.`);
    response.error = true;
    return response;
  }

  response.numPlayers = dbResponse.count;
  return response;
}

module.exports.getWinnerOpaqueUserIDFromGameState = function(gameState) {
  const winnerOpaqueUserID = gameState && gameState.gameWinnerMetadata && gameState.gameWinnerMetadata.opaqueUserID;
  return winnerOpaqueUserID == null ? constants.STR_NULL : winnerOpaqueUserID;
}

module.exports.getNumPlayersFromGameState = function(gameState) {
  if (gameState != null && gameState.playerStates != null) {
      return gameState.playerStates.length;
  }
  return constants.INT_NULL;
}

module.exports.userIsWinnerOfGameState = function(opaqueUserID, gameState) {
  const winnerOpaqueUserID = gameState && gameState.gameWinnerMetadata && gameState.gameWinnerMetadata.opaqueUserID;
  if (opaqueUserID == null || winnerOpaqueUserID == null) {
    return false;
  }
  return opaqueUserID == winnerOpaqueUserID;
}