const db = require("./dynamoDBHelpers.js");
const commonHelpers = require("./commonHelpers");
const constants = require("./constants.js");
const logger = require("../logger.js");

module.exports.migrateGameRecordFromLiveToHistory = async function(gameRecord, result) {
    // Shoot off metrics call but don't await on it
    module.exports.addToGameHistoryTable(gameRecord, result);

    return await module.exports.removeFromLiveGames(gameRecord.gameID);
}

module.exports.addToGameHistoryTable = async function(gameRecord) {
    const gameState = gameRecord.storedState;

    const liveGameHistoryData = {
        gameID: gameRecord.gameID,
        channelID: gameRecord.channelID,
        numPlayers: commonHelpers.getNumPlayersFromGameState(gameState),
        winnerOpaqueUserID: commonHelpers.getWinnerOpaqueUserIDFromGameState(gameState),
        startTime: gameRecord.createdAt,
        endTime: gameRecord.updatedAt
    }

    const putGameHistoryResponse = await db.dynamoPut(db.DynamoTables.GameHistory, liveGameHistoryData);

    if (!putGameHistoryResponse.success) {
        logger.error(`An error occurred while adding gameID: ${gameRecord.gameID} to the GameHistory table.`);
        return false;
    }

    return true;
}

module.exports.removeFromLiveGames = async function(gameID) {
    const key = {
        gameID: gameID
    };

    const deleteLiveGameResponse = await db.dynamoDelete(db.DynamoTables.LiveGames, key);

    if (!deleteLiveGameResponse.success) {
        logger.error(`An error occurred while removing game: ${gameID} from the LiveGames table.`);
        return false;
    }

    return true;
}


module.exports.migratePlayerIDFromLiveToHistory = async function(channelID, opaqueUserID, result = null) {
    // Get player record
    const playerResponse = await db.dynamoGet(db.DynamoTables.LivePlayers, {
        channelID: channelID,
        opaqueUserID: opaqueUserID
    });
    
    if (!playerResponse.success) {
        logger.error(`An error occurred when reading from LivePlayers for channelID: ${channelID}, opaquerUserID: ${opaqueUserID}.`);
        return false;
    }
    if (!playerResponse.itemFound) {
        logger.info(`Could not find player in LivePlayers to migrate to PlayerHistory for channelID: ${channelID}, opaquerUserID: ${opaqueUserID}. Assuming player already left.`);
        return true;
    }

    const playerRecord = playerResponse.item;

    return (await module.exports.migratePlayerRecordFromLiveToHistory(playerRecord, result));
}

async function determineGameResultForLivePlayer(playerRecord){
    const opaqueUserID = playerRecord.opaqueUserID;
    const gameID = playerRecord.gameID;

    // Get the game the player was in (if any)
    let result = constants.UNEXPECTED_RESULT;
    if (gameID != constants.NO_GAME_ID) {
        const gameResponse = await db.dynamoGet(db.DynamoTables.LiveGames, {
            gameID: gameID,
        });
        if (!gameResponse.success) {
            logger.error(`An error occurred when looking for ${opaqueUserID}'s game: ${gameID} in LiveGames.`);
        } else if (!gameResponse.itemFound) {
            logger.error(`${opaqueUserID}'s game: ${gameID} didn't exist in LiveGames.`);
        } else {
            const isWinner = commonHelpers.userIsWinnerOfGameState(opaqueUserID, gameResponse.item.storedState);
            result = isWinner ? constants.GAME_END_WON : constants.GAME_END_LOST;
        }
    }

    return result;
}

module.exports.migratePlayerRecordFromLiveToHistory = async function(playerRecord, result = null) {
    // Shoot off metrics call but don't await on it
    module.exports.addToPlayerHistoryTable(playerRecord, result);

    return await module.exports.removeFromLivePlayers(playerRecord.channelID, playerRecord.opaqueUserID);
}

module.exports.addToPlayerHistoryTable = async function(playerRecord, result) {
    result = result || await determineGameResultForLivePlayer(playerRecord);

    const livePlayerHistoryData = {
        channelID: playerRecord.channelID,
        opaqueUserID: playerRecord.opaqueUserID,
        gameID: playerRecord.gameID,
        userID: playerRecord.userID,
        startTime: playerRecord.createdAt,
        endTime: commonHelpers.getISOTimestamp(),
        result: result
    }

    const newPlayerHistoryResponse = await db.dynamoPut(db.DynamoTables.PlayerHistory, livePlayerHistoryData);

    if (!newPlayerHistoryResponse.success) {
        logger.error(`An error occurred while adding channelID: ${playerRecord.channelID}, opaqueUserID: ${playerRecord.opaqueUserID} to the PlayerHistory table.`);
        return false;
    }
    return true;
}

module.exports.removeFromLivePlayers = async function(channelID, opaqueUserID) {
    const key = {
        channelID: channelID,
        opaqueUserID: opaqueUserID
    };

    const deleteLivePlayerResponse = await db.dynamoDelete(db.DynamoTables.LivePlayers, key);

    if (!deleteLivePlayerResponse.success) {
        logger.error(`An error occurred while removing player with channelID: ${channelID}, opaqueUserID: ${opaqueUserID} from the LivePlayers table.`);
        return false;
    }

    return true;
}