const AWS = require("aws-sdk")
const constants = require('./constants.js');
const commonHelpers = require("./commonHelpers.js");
const logger = require("../logger.js");

const dynamo = new AWS.DynamoDB.DocumentClient({
    region: constants.AWS_REGION
});

module.exports.DynamoTables = {
    LiveSessions: "LiveSessions",
    LiveGames: "LiveGames",
    LivePlayers: "LivePlayers",
    LivePlayerInputs: "LivePlayerInputs",
    SessionHistory: "SessionHistory",
    GameHistory: "GameHistory",
    PlayerHistory: "PlayerHistory",
    WhitelistedChannels: "WhitelistedChannels"
};

/** Stores the given item in the specified dynamoDB table. 
 * Returns true for success, false for failure. */
module.exports.dynamoPut = function(tableName, item) {
    const now = commonHelpers.getISOTimestamp();

    item.createdAt = now
    item.updatedAt = now;

    return dynamo.put({
        TableName: tableName,
        Item: item,
    }).promise().then((response) => {
        return {success: true};
    }).catch((error) => {
        logger.error(`An error occurred when writing to table ${tableName}: ${error}`);
        return {success: false};
    })
}

/** Deletes the item specified by the given key from the specified dynamoDB table. 
 * Returns true for success, false for failure. */
module.exports.dynamoDelete = function(tableName, key) {
    return dynamo.delete({
        TableName: tableName,
        Key: key,
    }).promise().then((response) => {
        return {success: true};
    }).catch((error) => {
        logger.error(`An error occurred when deleting from table ${tableName}: ${error}`);
        return {success: false};
    })
}


/** Gets the item specified by the given key from the specified dynamoDB table. 
 * Returns the item on success, or false on failure. */
module.exports.dynamoGet = function(tableName, key) {
    return dynamo.get({
        TableName: tableName,
        Key: key,
        ConsistentRead: true,
    }).promise().then((response) => {
        let itemFound = (typeof response.Item !== "undefined");
        return {item: response.Item, itemFound: itemFound, success: true};
    }).catch((error) => {
        logger.error(`An error occurred when reading from table ${tableName}: ${error}`);
        return {success: false};
    })
}

/** Scans the specified dynamoDB table using the provided filter expression and attributes. 
 * Returns the item on success, or false on failure. */
module.exports.dynamoScan = function(tableName, filterExpression = null, expressionAttributeValues = null) {
    params = { TableName: tableName, ConsistenRead: true };
    if (filterExpression != null) {
        params.FilterExpression = filterExpression;
        params.ExpressionAttributeValues = expressionAttributeValues;
    }
    return dynamo.scan(params).promise().then((response) => {
        return {
            items: response.Items, 
            count: response.Count, 
            scannedCount: response.ScannedCount, 
            success: true
        };
    }).catch((error) => {
        logger.error(`An error occurred when scanning table ${tableName}: ${error}`);
        return {success: false};
    })
}

/** Updates the item specified by the given key from the specified dynamoDB table using the provided parameters. 
 * Returns the item on success, or false on failure. */
module.exports.dynamoUpdate = function(tableName, key, updatedValues, conditionExpression = null) {
    let updateExpression = "set ";
    let expressionAttributeValues = {};

    Object.keys(updatedValues).forEach((key) => {
        updateExpression += `${key} = :${key}, `;
        expressionAttributeValues[":" + key] = updatedValues[key];
    })

    updateExpression += "updatedAt = :updatedAt";
    expressionAttributeValues[":updatedAt"] = commonHelpers.getISOTimestamp();

    params = { 
        TableName: tableName,
        Key: key,
        UpdateExpression: updateExpression,
        ExpressionAttributeValues: expressionAttributeValues, 
    };

    if (conditionExpression != null) {
        params.ConditionExpression = conditionExpression;
    }

    return dynamo.update(params).promise().then((response) => {
        return {success: true};
    }).catch((error) => {
        logger.error(`An error occurred when updating table ${tableName}: ${error}`);
        return {success: false};
    })
}