const db = require("./dynamoDBHelpers.js");
const uuidv1 = require('uuid/v1');
const constants = require("./constants.js");
const twitchClient = require("../clients/twitchClient.js");
const GameEngine = require("../gameEngine.js");
const logger = require("../logger.js");
const dataMigrationHelpers = require("./dataMigrationHelpers.js");

/** Generates a gameID. This uses both current timestamp (uuidv1) and the game number to ensure uniqueness. */
function generateGameID() {
    return uuidv1();
}

/**
 * Updates an entry in the LivePlayers table using the channelID and opaqueUserID. 
 * The updatedValues is a anonymous object with key value pairs denoting the field to update and the
 * new value for that field. 
 */
async function updateLivePlayer(channelID, opaqueUserID, updatedValues) {
    const key = {
        channelID: channelID,
        opaqueUserID: opaqueUserID
    };

    const dbResponse = await db.dynamoUpdate(
        db.DynamoTables.LivePlayers,
        key,
        updatedValues
    );

    return dbResponse.success;
}

/**
 * Generates the initial states for each player in the match. If a user ID is available, we translate the ID
 * into the user's Twitch display name.
 */
async function generatePlayersMetadata(players) {
    const playersMetadata = [];
    const userIDsToDisplayName = (await twitchClient.translateUserIDsToDisplayNames(players)).result;

    players.forEach((player) => {
        const playerMetadata = {opaqueUserID: player.opaqueUserID, username: "Anonymous"};

        if (player.userID in userIDsToDisplayName) {
            playerMetadata.username = userIDsToDisplayName[player.userID];
        }

        playersMetadata.push(playerMetadata);
    });

    return playersMetadata;
}

/**
 * Removes all matchmaking players from the specified channel and records a PlayerHistory entry for them. 
 */
module.exports.sessionEndCancelMatchmakingForChannel = async function(channelID) {
    const getPlayersMatchmakingResponse = await module.exports.getPlayersMatchmaking(channelID);

    if (!getPlayersMatchmakingResponse.success) {
        logger.error(`Unable to obtain the players matchmaking for channel ${channelID}.`);
        return false;
    }

    if (getPlayersMatchmakingResponse.count > 0) {
        const migratePlayersPromises = [];
        getPlayersMatchmakingResponse.items.forEach((playerRecord) => {
            migratePlayersPromises.push(dataMigrationHelpers.migratePlayerRecordFromLiveToHistory(playerRecord, constants.MATCHMAKING_CANCELLED_BY_STREAMER));
        })
        const migratePlayersResult = await Promise.all(migratePlayersPromises);
        if (migratePlayersResult.includes(false)) {
            logger.error(`An error occurred while cancelling matchmaking for users due to session end on channel: ${channelID}`);
            return false;
        }
    }

    return true;
}

/** Sets the automatic match start timestamp for the given channel to the specified timestamp.
 * Returns true for success, false for failure. */
module.exports.setAutomaticMatchStartTimestamp = async function (channelID, timestamp) {
    const key = {
        channelID: channelID,
    };

    const updatedValues = {automaticMatchStartTimestamp: timestamp}

    const dbResponse = await db.dynamoUpdate(
        db.DynamoTables.LiveSessions,
        key,
        updatedValues
    );
    
    return dbResponse.success;
}

/** Removes the specified user from the matchmaking pool for the provided channel and puts them in the specified game.
 * Returns true for success, false for failure. */
module.exports.joinGame = function(channelID, opaqueUserID, gameID) {
    return updateLivePlayer(channelID, opaqueUserID, {matchmakingStartTimestamp: constants.NULL_TIMESTAMP, gameID: gameID});
};

/** Returns the users currently in the matchmaking pool (looking for a match). */
module.exports.getPlayersMatchmaking = function(channelID) {
    let filterExpression = 'channelID = :id AND matchmakingStartTimestamp <> :t';
    let expressionAttributeValues = {':id' : channelID, ':t' : constants.NULL_TIMESTAMP};
    return db.dynamoScan(db.DynamoTables.LivePlayers, filterExpression, expressionAttributeValues);
}

/** Returns all channels currently running the game. */
module.exports.getLiveChannels = function() {
    return db.dynamoScan(db.DynamoTables.LiveSessions);
}

/** Removes the given players from the matchmaking pool and starts a game for them. */
module.exports.startGameForPlayers = async function(channelID, players) {
    const gameID = generateGameID();
    const playersMetadata = await generatePlayersMetadata(players);
    const initializeGamePromise = GameEngine.initializeGame(channelID, gameID, playersMetadata);

    const joinGamePromises = [];
    players.forEach(player => {
        joinGamePromises.push(module.exports.joinGame(channelID, player.opaqueUserID, gameID));
    });

    await Promise.all([...joinGamePromises, initializeGamePromise]);
}