const JWT = require("jsonwebtoken");
const request = require("request-promise-native")
const AWS = require('aws-sdk');
const constants = require('./constants.js');
const logger = require("../logger.js");

// extension secret vars
const SECRET_NAME = "PrimeExtensionSecret";
var secret = null;

// Twitch PubSub broadcasting vars
const CLIENT_ID = "d4uvtfdr04uq6raoenvj7m86gdk16v";
const SEND_EXPIRATION_TIME = 120 * 1000;
const SEND_PERMISSIONS = {"send": ["*"]};
const PUBSUB_SEND_URL = "https://api.twitch.tv/extensions/message/:channelID";

// Retrieve the extension secret on application start
var secretsManagerClient = new AWS.SecretsManager({
    region: constants.AWS_REGION
});
secretsManagerClient.getSecretValue({SecretId: SECRET_NAME}, function(err, data) {
    if (err) {
        logger.error(`Error getting extension secret: ${err.code}`);
        return;
    }
    else {
        // Depending on whether the secret is a string or binary, one of data.SecretString and data.SecretBinary will be populated.
        if ('SecretString' in data) {
            secret = Buffer.from(data.SecretString, "base64");
        } else {
            secret = new Buffer(data.SecretBinary, 'base64');
        }
    }
});

module.exports.getExtensionSecret = function () {
    return secret;
}

module.exports.broadcast = function(parameters) {
    let uri = PUBSUB_SEND_URL.replace(":channelID", parameters.channelID)
    let authorization = JWT.sign({
        "role": "external",
        "pubsub_perms": SEND_PERMISSIONS,
        "exp": Date.now() + SEND_EXPIRATION_TIME,
        "channel_id": parameters.channelID
    }, secret)

    let body = {
        "targets": ["broadcast"],
        "message": JSON.stringify(parameters.message),
        "content_type": "application/json",
    }

    let headers = {
        "Content-Type": "application/json",
        "Authorization": `Bearer ${authorization}`,
        "Client-Id": CLIENT_ID,
    }

    return request({
        "resolveWithFullResponse": true,
        "uri": uri,
        "method": "POST",
        "headers": headers,
        "json": true,
        "body": body,
    }).catch((err) => {
        logger.error(`Error broadcasting via Twitch PubSub: ${err}`);
    })
}